"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthRouter = void 0;
const express_1 = require("express");
/**
 * Generic authentication router that can work with any authenticatable entity
 * @template T - Type of entity that extends IAuthenticatableEntity
 * @template R - Type of registration parameters that extends IRegisterParams
 */
class AuthRouter {
    constructor(controller, authMiddleware) {
        this.path = '/auth';
        this.router = (0, express_1.Router)();
        this.controller = controller;
        this.authMiddleware = authMiddleware;
        this.initRoutes();
    }
    /**
     * Initializes all authentication routes
     */
    initRoutes() {
        /**
         * Perform a new registration
         * @method POST
         * @param path - /auth/register
         */
        this.router.post('/register', this.controller.register);
        /**
         * Perform a log in
         * @method POST
         * @param path - /auth/login
         */
        this.router.post('/login', this.controller.login);
        /**
         * Requests password reset
         * @method POST
         * @param path - /auth/forgot-password
         */
        this.router.post('/forgot-password', this.controller.forgotPassword);
        /**
         * Resets password using code
         * @method POST
         * @param path - /auth/reset-password
         */
        this.router.post('/reset-password', this.controller.resetPassword);
        /**
         * View profile
         * @method GET
         * @param path - /auth/profile
         */
        this.router.get('/profile', this.authMiddleware, this.controller.viewProfile);
        /**
         * Edit profile
         * @method PUT
         * @param path - /auth/profile
         */
        this.router.put('/profile', this.authMiddleware, this.controller.editProfile);
    }
    /**
     * Returns the router with all routes
     */
    getRoutes() {
        return this.router;
    }
}
exports.AuthRouter = AuthRouter;
