"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CategoryRouter = void 0;
const express_1 = require("express");
/**
 * Router class for category endpoints.
 * Maps HTTP routes to CategoryController methods.
 */
class CategoryRouter {
    constructor(controller, authMiddleware, permissionsMiddleware) {
        this.path = '/admin/categories';
        this.controller = controller;
        this.authMiddleware = authMiddleware;
        this.permissionsMiddleware = permissionsMiddleware;
        this.router = (0, express_1.Router)();
        this.initRoutes();
    }
    /**
     * Initializes all routes for the category controller
     * Maps endpoints to controller methods
     */
    initRoutes() {
        /**
         * Unauthenticated users can view categories
        **/
        /**
         * Lists all categorys
         * @method GET
         * @param path - /categories
         */
        this.router.get('/', this.controller.listCategories);
        /**
         * Finds a category by ID
         * @method GET
         * @param path - /categories/:id
         */
        this.router.get('/:id', this.controller.getCategoryById);
        // Apply the auth middlewares
        this.router.use(this.authMiddleware);
        /**
         * Authenticated users with the correct permissions can manipulate categories
        **/
        /**
         * Creates a new category
         * @method POST
         * @param path - /categories
         */
        this.router.post('/', this.controller.createCategory);
        /**
         * Updates a category by ID
         * @method PUT
         * @param path - /categories/:id
         */
        this.router.put('/:id', this.controller.updateCategory);
        /**
         * Deletes a category by ID
         * @method DELETE
         * @param path - /categories/:id
         */
        this.router.delete('/:id', this.controller.deleteCategory);
    }
    /**
     * Get the router with all routes
     * @returns Express Router instance
     */
    getRoutes() {
        return this.router;
    }
}
exports.CategoryRouter = CategoryRouter;
