"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DisciplineRouter = void 0;
const express_1 = require("express");
/**
 * Router class for discipline endpoints.
 * Maps HTTP routes to DisciplineController methods.
 */
class DisciplineRouter {
    constructor(controller, authMiddleware, permissionsMiddleware) {
        this.path = '/admin/disciplines';
        this.controller = controller;
        this.authMiddleware = authMiddleware;
        this.permissionsMiddleware = permissionsMiddleware;
        this.router = (0, express_1.Router)();
        this.initRoutes();
    }
    /**
     * Initializes all routes for the discipline controller
     * Maps endpoints to controller methods
     */
    initRoutes() {
        /**
         * List all disciplines
         * @method GET
         * @param path - /disciplines
         */
        this.router.get('/', this.controller.listDisciplines);
        /**
         * Finds a discipline by ID
         * @method GET
         * @param path - /disciplines/:id
         */
        this.router.get('/:id', this.controller.getDisciplineById);
        // Apply the auth middlewares
        this.router.use(this.authMiddleware);
        /**
         * Authenticated users with the correct permissions can manipulate disciplines
        **/
        /**
         * Creates a new a discipline
         * @method POST
         * @param path - /disciplines
         */
        this.router.post('/:id', this.controller.createDiscipline);
        /**
         * Update a discipline by ID
         * @method PUT
         * @param path - /disciplines/:id
         */
        this.router.put('/:id', this.controller.updateDiscipline);
        /**
         * Deletes a discipline by ID
         * @method DELETE
         * @param path - /disciplines/:id
         */
        this.router.delete('/:id', this.controller.deleteDiscipline);
    }
    /**
     * Get the router with all routes
     * @returns Express Router instance
     */
    getRoutes() {
        return this.router;
    }
}
exports.DisciplineRouter = DisciplineRouter;
