"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RolesRouter = void 0;
const express_1 = require("express");
/**
 * Router class for Role endpoints.
 * Maps HTTP routes to RoleController methods.
 */
class RolesRouter {
    constructor(rolesController, authMiddleware, permissionMiddleware) {
        this.path = '/admin/roles';
        this.controller = rolesController;
        this.authMiddleware = authMiddleware;
        this.permissionMiddleware = permissionMiddleware;
        this.router = (0, express_1.Router)();
        this.initRoutes();
    }
    /**
     * Initialize all routes for the controller
     * Maps endpoints to controller methods
     */
    initRoutes() {
        /*
         * User auth middleware
         * Authenticated users can view existing permissions
         */
        this.router.use(this.authMiddleware);
        /**
         * Lists all roles
         * @method GET
         * @param path - /roles
         */
        this.router.get("/", this.controller.getRoles);
        /**
         * Get a role by ID
         * @method GET
         * @param path - /roles/:id
         */
        this.router.get("/:id", this.controller.getRoleById);
        /*
         * Permission middleware
         * Authorized users can create, edit, and delete permissions
         */
        this.router.use(this.permissionMiddleware);
        /**
         * Create a new role
         * @method POST
         * @param path - /roles
         */
        this.router.post("/", this.controller.createRole);
        /**
         * Update a role by ID
         * @method PUT
         * @param path - /roles:id
         */
        this.router.put("/:id", this.controller.updateRole);
        /**
         * Delete a role by ID
         * @method DELETE
         * @param path - /roles:id
         */
        this.router.delete("/:id", this.controller.deleteRole);
    }
    /**
     * Get the router with all routes
     * @returns Express Router instance
     */
    getRoutes() {
        return this.router;
    }
}
exports.RolesRouter = RolesRouter;
