"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TeacherRouter = void 0;
const express_1 = require("express");
/**
 * Router class for teachers endpoints.
 * Maps HTTP routes to TeacherController methods.
 */
class TeacherRouter {
    constructor(controller, authRouter, authMiddleware) {
        this.path = "/app/teachers";
        this.controller = controller;
        this.authRouter = authRouter;
        this.authMiddleware = authMiddleware;
        this.router = (0, express_1.Router)();
        this.initRoutes();
    }
    /**
     * Initializes all routes for the teacher router
     * Maps endpoints to controller methods
     */
    initRoutes() {
        /**
         * Apply the authentication routes
         */
        this.router.use(this.authRouter.path, this.authRouter.getRoutes());
        /**
         * Apply the auth middleware
         */
        this.router.use(this.authMiddleware);
        /**
         * Creates a new teacher
         * @method POST
         * @param path - /teachers
         */
        this.router.post("/", this.controller.createTeacher);
        /**
         * List all teachers
         * @method GET
         * @param path - /teachers
         */
        this.router.get("/", this.controller.listTeachers);
        /**
         * List teacher's classes
         * @method GET
         * @param path - /teachers/classes
         */
        this.router.get("/classes", this.controller.getClasses);
        /**
         * Find one class by ID
         * @method GET
         * @param path - /teachers/classes/:id
         */
        this.router.get("/classes/:id", this.controller.getClassById);
        /**
         * Finish one class by ID
         * @method GET
         * @param path - /teachers/classes/:id/finish
         */
        this.router.post("/classes/:id/finish", this.controller.finishClass);
    }
    /**
     * Get the router with all routes
     * @returns Express Router instance
     */
    getRoutes() {
        return this.router;
    }
}
exports.TeacherRouter = TeacherRouter;
