"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.allAuthSchemas = void 0;
const zod_1 = require("zod");
const email_schema_1 = require("./shared/email.schema");
const password_schema_1 = require("./shared/password.schema");
const user_schema_1 = require("./user.schema");
const student_schema_1 = require("./student.schema");
const teacher_schema_1 = require("./teacher.schema");
const loginBodySchema = zod_1.z.object({
    email: email_schema_1.emailValidation,
    password: password_schema_1.passwordValidation,
});
const registerBodySchema = zod_1.z.object({
    email: email_schema_1.emailValidation,
    password: password_schema_1.passwordValidation,
}).passthrough();
const forgotPasswordBodySchema = zod_1.z.object({
    email: email_schema_1.emailValidation,
});
const resetPasswordBodySchema = zod_1.z.object({
    email: email_schema_1.emailValidation,
    code: zod_1.z
        .string()
        .trim()
        .length(6, 'Verification code must have 6 characters.'),
    newPassword: password_schema_1.passwordValidation,
});
/**
 * Collection of schemas responsible for validating authentication requests.
 */
const authSchemas = {
    register: zod_1.z.object({
        body: registerBodySchema,
    }),
    login: zod_1.z.object({
        body: loginBodySchema,
    }),
    forgotPassword: zod_1.z.object({
        body: forgotPasswordBodySchema,
    }),
    resetPassword: zod_1.z.object({
        body: resetPasswordBodySchema,
    }),
};
/**
 * Schemas for specific entity registration and profile updates.
 * These leverage existing entity-specific schemas.
 */
exports.allAuthSchemas = {
    userAuthSchemas: Object.assign(Object.assign({}, authSchemas), { register: user_schema_1.userSchemas.createUser, update: zod_1.z.object({
            body: user_schema_1.userSchemas.updateUser.shape.body,
        }) }),
    // Student
    studentAuthSchemas: Object.assign(Object.assign({}, authSchemas), { register: student_schema_1.studentSchemas.createStudent, update: zod_1.z.object({
            body: student_schema_1.studentSchemas.updateStudent.shape.body,
        }) }),
    // Teacher
    teacherAuthSchemas: Object.assign(Object.assign({}, authSchemas), { register: teacher_schema_1.teacherSchemas.createTeacher, update: zod_1.z.object({
            body: teacher_schema_1.teacherSchemas.updateTeacher.shape.body,
        }) }),
};
