"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.categoryNameSchema = exports.categoryIdsSchema = exports.categorySchemas = exports.categorySchema = void 0;
const zod_1 = require("zod");
const uuidv7_schema_1 = require("./shared/uuidv7.schema");
/**
 * Shared fields for category-related payloads
 */
exports.categorySchema = {
    name: zod_1.z
        .string()
        .trim()
        .min(1, 'Category name cannot be empty.')
        .max(100, 'Category name cannot exceed 100 characters.')
        .regex(/^[a-zA-Z0-9À-ÿ\s\-_&]+$/, 'Category name can only contain letters, numbers, spaces, hyphens, underscores, and ampersands.')
        .refine((name) => {
        // Check for consecutive special characters
        return !/(\s{2,}|_{2,}|-{2,})/.test(name);
    }, {
        message: 'Category name cannot contain consecutive special characters or spaces.',
    }),
    description: zod_1.z
        .string()
        .trim()
        .min(1, 'Category description cannot be empty.')
        .max(500, 'Category description cannot exceed 500 characters.')
        .optional(),
};
/**
 * Schema for creating a new category
 */
const createCategoryBodySchema = zod_1.z.object({
    name: exports.categorySchema.name,
    description: exports.categorySchema.description,
});
/**
 * Schema for updating an existing category
 */
const updateCategoryBodySchema = zod_1.z
    .object({
    name: exports.categorySchema.name.optional(),
    description: exports.categorySchema.description,
})
    .superRefine((data, ctx) => {
    if (Object.keys(data).length === 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Provide at least one field to update.',
            path: [],
        });
    }
    // Validate that if description is provided, it's not empty
    if (data.description !== undefined && data.description.trim() === '') {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Description cannot be empty if provided.',
            path: ['description'],
        });
    }
    // Validate that name doesn't contain only special characters when provided
    if (data.name !== undefined) {
        const nameWithoutSpecialChars = data.name.replace(/[^a-zA-Z0-9À-ÿ]/g, '');
        if (nameWithoutSpecialChars.length === 0) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: 'Category name must contain at least one letter or number.',
                path: ['name'],
            });
        }
    }
});
/**
 * Collection of schemas responsible for validating all category module requests.
 */
exports.categorySchemas = {
    createCategory: zod_1.z.object({
        body: createCategoryBodySchema,
    }),
    getCategoryById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
    updateCategory: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: updateCategoryBodySchema,
    }),
    deleteCategory: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
};
/**
 * Schema that validates a non-empty list of category IDs for bulk operations
 */
exports.categoryIdsSchema = zod_1.z
    .array(uuidv7_schema_1.uuidV7Schema)
    .min(1, 'Provide at least one category ID')
    .max(100, 'Cannot process more than 100 categories at once');
/**
 * Schema for category name uniqueness check
 */
exports.categoryNameSchema = zod_1.z.object({
    name: exports.categorySchema.name,
    exclude_id: uuidv7_schema_1.uuidV7Schema.optional(), // For update operations, exclude current category
});
