"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.classSchemas = exports.getClassesQuerySchema = void 0;
const zod_1 = require("zod");
const uuidv7_schema_1 = require("./shared/uuidv7.schema");
const simpleString_schema_1 = require("./shared/simpleString.schema");
const date_schema_1 = require("./shared/date.schema");
/**
 * Shared fields for class-related payloads
 */
const classSchema = {
    topic_id: uuidv7_schema_1.uuidV7Schema,
    teacher_id: uuidv7_schema_1.uuidV7Schema,
    main_student_id: uuidv7_schema_1.uuidV7Schema,
    additional_students: zod_1.z
        .array(uuidv7_schema_1.uuidV7Schema)
        .max(20, 'Cannot exceed 20 additional students.')
        .optional(),
    link: simpleString_schema_1.stringSchemas.url.optional(),
    type: zod_1.z.enum(['individual', 'group']),
    start_time: date_schema_1.dateValidation.isoDateSchema
        .transform(val => new Date(val))
        .pipe(zod_1.z.date()),
    end_time: date_schema_1.dateValidation.isoDateSchema
        .transform(val => new Date(val))
        .pipe(zod_1.z.date()),
    status: zod_1.z.enum(['scheduled', 'canceled']),
    payment_status: zod_1.z.enum(['pending', 'paid', 'refunded']),
};
/**
 * Schema for creating a new class
 */
const createClassBodySchema = zod_1.z
    .object({
    topic_id: classSchema.topic_id,
    teacher_id: classSchema.teacher_id,
    main_student_id: classSchema.main_student_id,
    additional_students: classSchema.additional_students.optional(),
    start_time: classSchema.start_time,
    end_time: classSchema.end_time.optional(),
})
    .superRefine((data, ctx) => {
    // Validate that end_time > start_time
    if (data.end_time && data.end_time <= data.start_time) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'end_time must be after start_time.',
            path: ['end_time'],
        });
    }
});
/**
 * Schema for updating an existing class
 */
const updateClassBodySchema = zod_1.z
    .object({
    topic_id: classSchema.topic_id.optional(),
    teacher_id: classSchema.teacher_id.optional(),
    additional_students: classSchema.additional_students.optional(),
    start_time: classSchema.start_time.optional(),
    end_time: classSchema.end_time.optional(),
    status: classSchema.status.optional(),
    payment_status: classSchema.payment_status.optional(),
})
    .superRefine((data, ctx) => {
    if (Object.keys(data).length === 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Provide at least one field to update.',
            path: [],
        });
    }
    // Check end_time > start_time if both exist
    if (data.start_time && data.end_time && data.end_time <= data.start_time) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'end_time must be after start_time.',
            path: ['end_time'],
        });
    }
});
/**
 * Query parameters for filtering classes
 */
exports.getClassesQuerySchema = zod_1.z.object({
    topic_id: uuidv7_schema_1.uuidV7Schema.optional(),
    teacher_id: uuidv7_schema_1.uuidV7Schema.optional(),
    student_id: uuidv7_schema_1.uuidV7Schema.optional(),
    type: zod_1.z.enum(['individual', 'group']).optional(),
    status: zod_1.z.enum(['scheduled', 'canceled', 'completed']).optional(),
    payment_status: zod_1.z.enum(['pending', 'paid', 'refunded']).optional(),
});
/**
 * Collection of schemas responsible for validating all class module requests.
 */
exports.classSchemas = {
    createClass: zod_1.z.object({
        body: createClassBodySchema,
    }),
    getClassById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
    getClasses: zod_1.z.object({
        query: exports.getClassesQuerySchema,
    }),
    updateClass: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: updateClassBodySchema,
    }),
    deleteClass: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
};
