"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.disciplineSchemas = exports.disciplineSchema = void 0;
const zod_1 = require("zod");
const uuidv7_schema_1 = require("./shared/uuidv7.schema");
/**
 * Shared fields for discipline-related payloads
 */
exports.disciplineSchema = {
    name: zod_1.z
        .string()
        .trim()
        .min(1, 'Discipline name cannot be empty.')
        .max(100, 'Discipline name cannot exceed 100 characters.')
        .regex(/^[a-zA-Z0-9À-ÿ\s\-_&()]+$/, 'Discipline name can only contain letters, numbers, spaces, hyphens, underscores, ampersands, and parentheses.')
        .refine((name) => {
        // Check for consecutive special characters
        return !/(\s{2,}|_{2,}|-{2,}|\(\))/.test(name);
    }, {
        message: 'Discipline name cannot contain consecutive special characters, spaces, or empty parentheses.',
    }),
    description: zod_1.z
        .string()
        .trim()
        .min(1, 'Discipline description cannot be empty.')
        .max(500, 'Discipline description cannot exceed 500 characters.')
        .optional(),
    category_id: uuidv7_schema_1.uuidV7Schema,
};
/**
 * Schema for creating a new discipline
 */
const createDisciplineBodySchema = zod_1.z.object({
    name: exports.disciplineSchema.name,
    description: exports.disciplineSchema.description,
    category_id: exports.disciplineSchema.category_id,
});
/**
 * Query parameters for filtering disciplines
 */
const listDisciplinesQuerySchema = zod_1.z.object({
    category_id: exports.disciplineSchema.category_id.optional(),
});
/**
 * Schema for updating an existing discipline
 */
const updateDisciplineBodySchema = zod_1.z
    .object({
    name: exports.disciplineSchema.name.optional(),
    description: exports.disciplineSchema.description,
    category_id: exports.disciplineSchema.category_id.optional(),
})
    .superRefine((data, ctx) => {
    if (Object.keys(data).length === 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Provide at least one field to update.',
            path: [],
        });
    }
    // Validate that if description is provided, it's not empty
    if (data.description !== undefined && data.description.trim() === '') {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Description cannot be empty if provided.',
            path: ['description'],
        });
    }
    // Validate that name doesn't contain only special characters when provided
    if (data.name !== undefined) {
        const nameWithoutSpecialChars = data.name.replace(/[^a-zA-Z0-9À-ÿ]/g, '');
        if (nameWithoutSpecialChars.length === 0) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: 'Discipline name must contain at least one letter or number.',
                path: ['name'],
            });
        }
    }
    // Validate that category_id is a valid UUID v7 if provided
    if (data.category_id !== undefined) {
        if (data.category_id.trim() === '') {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: 'Category ID cannot be empty if provided.',
                path: ['category_id'],
            });
        }
    }
});
/**
 * Collection of schemas responsible for validating all discipline module requests.
 */
exports.disciplineSchemas = {
    createDiscipline: zod_1.z.object({
        body: createDisciplineBodySchema,
    }),
    getDisciplineById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
    listDisciplines: zod_1.z.object({
        query: listDisciplinesQuerySchema,
    }),
    updateDiscipline: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: updateDisciplineBodySchema,
    }),
    deleteDiscipline: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
};
