"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.roleSchemas = exports.roleSchema = void 0;
const zod_1 = require("zod");
const uuidv7_schema_1 = require("./shared/uuidv7.schema");
/**
 * Shared fields for role-related payloads.
 */
exports.roleSchema = {
    name: zod_1.z
        .string()
        .trim()
        .min(1, 'Role name cannot be empty.')
        .max(100, 'Role name cannot exceed 100 characters.')
        .regex(/^[a-zA-Z0-9_\- ]+$/, 'Role name can only contain letters, numbers, spaces, hyphens and underscores.'),
    description: zod_1.z
        .string()
        .trim()
        .min(1, 'Role description cannot be empty.')
        .max(500, 'Role description cannot exceed 500 characters.')
};
/**
 * Schema for creating a new role
 */
const createRoleBodySchema = zod_1.z.object({
    name: exports.roleSchema.name,
    description: exports.roleSchema.description,
});
/**
 * Schema for updating an existing role
 */
const updateRoleBodySchema = zod_1.z
    .object({
    name: exports.roleSchema.name.optional(),
    description: exports.roleSchema.description.optional(),
})
    .superRefine((data, ctx) => {
    if (Object.keys(data).length === 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Provide at least one field to update.',
            path: [],
        });
    }
    // Validate that if description is provided, it's not empty
    if (data.description !== undefined && data.description.trim() === '') {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Description cannot be empty if provided.',
            path: ['description'],
        });
    }
});
/**
 * Collection of schemas responsible for validating all role module requests.
 */
exports.roleSchemas = {
    createRole: zod_1.z.object({
        body: createRoleBodySchema,
    }),
    getRoleById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
    updateRole: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: updateRoleBodySchema,
    }),
    deleteRoleById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
};
