"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.studentSchemas = exports.studentSchema = void 0;
const zod_1 = require("zod");
const uuidv7_schema_1 = require("./shared/uuidv7.schema");
const simpleString_schema_1 = require("./shared/simpleString.schema");
const date_schema_1 = require("./shared/date.schema");
const password_schema_1 = require("./shared/password.schema");
const email_schema_1 = require("./shared/email.schema");
const cpf_schema_1 = require("./shared/cpf.schema");
const rg_schema_1 = require("./shared/rg.schema");
const telephone_schema_1 = require("./shared/telephone.schema");
const class_schema_1 = require("./class.schema");
/**
 * Utility function to validate credit card number
 */
function isValidCreditCard(cardNumber) {
    const cleaned = cardNumber.replace(/\s/g, '');
    return /^\d{13,19}$/.test(cleaned);
}
/**
 * Utility function to validate CVV
 */
function isValidCVV(cvv) {
    return /^\d{3,4}$/.test(cvv);
}
/**
 * Utility function to validate expiration date (MM/YY)
 */
function isValidExpiration(expiration) {
    if (!/^\d{2}\/\d{2}$/.test(expiration))
        return false;
    const [month, year] = expiration.split('/').map(Number);
    if (month < 1 || month > 12)
        return false;
    const currentYear = new Date().getFullYear() % 100;
    const currentMonth = new Date().getMonth() + 1;
    if (year < currentYear)
        return false;
    if (year === currentYear && month < currentMonth)
        return false;
    return true;
}
/**
 * Shared fields for student-related payloads
 */
exports.studentSchema = {
    full_name: simpleString_schema_1.stringSchemas.portugueseName,
    birth_date: date_schema_1.dateValidation.isoDateSchema
        .transform(val => new Date(val))
        .pipe(zod_1.z.date()
        .max(new Date(new Date().setFullYear(new Date().getFullYear() - 3)), 'Student must be at least 3 years old.')
        .min(new Date(new Date().setFullYear(new Date().getFullYear() - 100)), 'Student cannot be older than 100 years.')),
    gender: zod_1.z
        .enum(['male', 'female', 'other'])
        .optional(),
    email: email_schema_1.emailValidation,
    password: password_schema_1.passwordValidation,
    cpf: zod_1.z.object({
        value: cpf_schema_1.cpfValidation,
        upload_id: uuidv7_schema_1.uuidV7Schema.optional(),
    }),
    rg: zod_1.z.object({
        value: rg_schema_1.rgValidation,
        upload_id: uuidv7_schema_1.uuidV7Schema.optional(),
    }).optional(),
    telephone: telephone_schema_1.telephoneValidation,
    academic: zod_1.z.object({
        current_grade: zod_1.z
            .string()
            .trim()
            .min(1, 'Current grade cannot be empty.')
            .max(50, 'Current grade cannot exceed 50 characters.'),
        school: zod_1.z
            .string()
            .trim()
            .min(1, 'School cannot be empty if provided.')
            .max(100, 'School cannot exceed 100 characters.')
            .optional(),
        subjects: zod_1.z
            .array(zod_1.z.string().trim().min(1, 'Subject cannot be empty.'))
            .min(1, 'Provide at least one subject.')
            .max(10, 'Cannot exceed 10 subjects.'),
        difficulty_level: zod_1.z
            .enum(['beginner', 'intermediate', 'advanced']),
        main_goal: zod_1.z
            .string()
            .trim()
            .min(1, 'Main goal cannot be empty.')
            .max(200, 'Main goal cannot exceed 200 characters.'),
    }),
    class_preferences: zod_1.z.object({
        modality: zod_1.z.enum(['online', 'presencial', 'hibrida']),
        frequency: zod_1.z
            .string()
            .trim()
            .min(1, 'Frequency cannot be empty.')
            .max(50, 'Frequency cannot exceed 50 characters.'),
        available_times: zod_1.z
            .array(zod_1.z.object({
            day_of_week: zod_1.z
                .number()
                .int()
                .min(0, 'Day of week must be between 0 and 6.')
                .max(6, 'Day of week must be between 0 and 6.'),
            start_time: zod_1.z
                .number()
                .int()
                .min(0, 'Start time must be between 0 and 1439.')
                .max(1439, 'Start time must be between 0 and 1439.'),
            end_time: zod_1.z
                .number()
                .int()
                .min(0, 'End time must be between 0 and 1439.')
                .max(1439, 'End time must be between 0 and 1439.'),
        }))
            .min(1, 'Provide at least one available time.')
            .refine(times => {
            return times.every(time => time.end_time > time.start_time);
        }, 'End time must be after start time for all available times.'),
        preferred_teacher: uuidv7_schema_1.uuidV7Schema.optional(),
    }),
    guardian: zod_1.z.object({
        full_name: zod_1.z
            .string()
            .trim()
            .min(1, 'Guardian full name cannot be empty.')
            .max(200, 'Guardian full name cannot exceed 200 characters.'),
        birth_date: zod_1.z
            .string()
            .or(zod_1.z.date())
            .transform(val => new Date(val))
            .pipe(zod_1.z.date()
            .max(new Date(new Date().setFullYear(new Date().getFullYear() - 18)), 'Guardian must be at least 18 years old.')
            .min(new Date(new Date().setFullYear(new Date().getFullYear() - 100)), 'Guardian cannot be older than 100 years.')),
        telephone: telephone_schema_1.telephoneValidation,
        email: email_schema_1.emailValidation,
        relationship: zod_1.z
            .string()
            .trim()
            .min(1, 'Relationship cannot be empty.')
            .max(50, 'Relationship cannot exceed 50 characters.'),
        cpf: zod_1.z.object({
            value: cpf_schema_1.cpfValidation,
            upload_id: uuidv7_schema_1.uuidV7Schema.optional(),
        }),
        rg: zod_1.z.object({
            value: rg_schema_1.rgValidation,
            upload_id: uuidv7_schema_1.uuidV7Schema.optional(),
        }),
    }).optional(),
    financial: zod_1.z.object({
        credit_card_registered: zod_1.z.boolean(),
        card_info: zod_1.z.object({
            cardholder_name: zod_1.z
                .string()
                .trim()
                .min(1, 'Cardholder name cannot be empty.')
                .max(100, 'Cardholder name cannot exceed 100 characters.')
                .regex(/^[a-zA-ZÀ-ÿ\s']+$/, 'Cardholder name can only contain letters and spaces.'),
            card_number: zod_1.z
                .string()
                .trim()
                .refine(isValidCreditCard, 'Provide a valid credit card number.'),
            expiration: zod_1.z
                .string()
                .trim()
                .refine(isValidExpiration, 'Provide a valid expiration date (MM/YY).'),
            cvv: zod_1.z
                .string()
                .trim()
                .refine(isValidCVV, 'Provide a valid CVV (3 or 4 digits).'),
        }).optional(),
    }).optional(),
};
/**
 * Schema for creating a new student
 */
const createStudentBodySchema = zod_1.z.object({
    full_name: exports.studentSchema.full_name,
    birth_date: exports.studentSchema.birth_date,
    gender: exports.studentSchema.gender,
    email: exports.studentSchema.email,
    password: exports.studentSchema.password,
    cpf: exports.studentSchema.cpf,
    rg: exports.studentSchema.rg,
    telephone: exports.studentSchema.telephone,
    academic: exports.studentSchema.academic,
    class_preferences: exports.studentSchema.class_preferences,
    guardian: exports.studentSchema.guardian,
    financial: exports.studentSchema.financial,
});
/**
 * Schema for updating an existing student
 */
const updateStudentBodySchema = zod_1.z
    .object({
    birth_date: exports.studentSchema.birth_date.optional(),
    gender: exports.studentSchema.gender,
    cpf: exports.studentSchema.cpf.optional(),
    rg: exports.studentSchema.rg,
    telephone: exports.studentSchema.telephone.optional(),
    academic: exports.studentSchema.academic.optional(),
    class_preferences: exports.studentSchema.class_preferences.optional(),
    guardian: exports.studentSchema.guardian.optional(),
    financial: exports.studentSchema.financial.optional(),
})
    .superRefine((data, ctx) => {
    if (Object.keys(data).length === 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Provide at least one field to update.',
            path: [],
        });
    }
    // Validate that if academic is provided, all required fields are present
    if (data.academic && Object.keys(data.academic).length > 0) {
        if (!data.academic.current_grade || !data.academic.subjects || !data.academic.difficulty_level || !data.academic.main_goal) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: 'When updating academic info, provide all required fields: current_grade, subjects, difficulty_level, main_goal.',
                path: ['academic'],
            });
        }
    }
    // Validate that if class_preferences is provided, all required fields are present
    if (data.class_preferences && Object.keys(data.class_preferences).length > 0) {
        if (!data.class_preferences.modality || !data.class_preferences.frequency || !data.class_preferences.available_times) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: 'When updating class preferences, provide all required fields: modality, frequency, available_times.',
                path: ['class_preferences'],
            });
        }
    }
});
/**
 * Schema for query parameters when searching for teachers
 */
const getTeachersQuerySchema = zod_1.z.object({
    days_of_week: zod_1.z
        .any()
        .transform(val => {
        let result = [];
        if (val === null || val === undefined) {
            return val;
        }
        if (Array.isArray(val)) {
            result = val.map(item => String(item).replace(/['"]/g, ''));
        }
        else if (typeof val === 'string') {
            try {
                const parsed = JSON.parse(val);
                if (Array.isArray(parsed)) {
                    result = parsed.map(item => String(item).replace(/['"]/g, ''));
                }
                else {
                    result = [String(parsed).replace(/['"]/g, '')];
                }
            }
            catch (_a) {
                if (val.includes(',')) {
                    result = val.split(',').map(day => day.trim().replace(/['"]/g, ''));
                }
                else {
                    result = [val.replace(/['"]/g, '')];
                }
            }
        }
        else {
            result = [String(val).replace(/['"]/g, '')];
        }
        return result;
    })
        .pipe(zod_1.z.array(zod_1.z.string())
        .min(1, 'Provide at least one day of week.')
        .refine(days => {
        const validDays = ['0', '1', '2', '3', '4', '5', '6'];
        return days.every(day => validDays.includes(day));
    }, 'Days of week must be numbers between 0 and 6.'))
        .optional(),
    start_time: zod_1.z
        .string()
        .transform(val => parseInt(val, 10))
        .pipe(zod_1.z.number()
        .int()
        .min(0, 'Start time must be between 0 and 1439.')
        .max(1439, 'Start time must be between 0 and 1439.'))
        .optional(),
    end_time: zod_1.z
        .string()
        .transform(val => parseInt(val, 10))
        .pipe(zod_1.z.number()
        .int()
        .min(0, 'End time must be between 0 and 1439.')
        .max(1439, 'End time must be between 0 and 1439.'))
        .optional(),
    discipline_id: uuidv7_schema_1.uuidV7Schema.optional(),
})
    .refine(data => {
    if (data.start_time && data.end_time) {
        return data.end_time > data.start_time;
    }
    return true;
}, 'End time must be after start time.');
const listAdaptedStudentsQuery = zod_1.z.object({
    daysOfWeek: zod_1.z
        .any()
        .transform(val => {
        let result = [];
        if (val === null || val === undefined) {
            return val;
        }
        if (Array.isArray(val)) {
            result = val.map(item => String(item).replace(/['"]/g, ''));
        }
        else if (typeof val === 'string') {
            try {
                const parsed = JSON.parse(val);
                if (Array.isArray(parsed)) {
                    result = parsed.map(item => String(item).replace(/['"]/g, ''));
                }
                else {
                    result = [String(parsed).replace(/['"]/g, '')];
                }
            }
            catch (_a) {
                if (val.includes(',')) {
                    result = val.split(',').map(day => day.trim().replace(/['"]/g, ''));
                }
                else {
                    result = [val.replace(/['"]/g, '')];
                }
            }
        }
        else {
            result = [String(val).replace(/['"]/g, '')];
        }
        return result;
    })
        .pipe(zod_1.z.array(zod_1.z.string())
        .min(1, 'Provide at least one day of week.')
        .refine(days => {
        const validDays = ['0', '1', '2', '3', '4', '5', '6'];
        return days.every(day => validDays.includes(day));
    }, 'Days of week must be numbers between 0 and 6.'))
        .optional(),
    startMinutes: zod_1.z
        .string()
        .transform(val => parseInt(val, 10))
        .pipe(zod_1.z.number()
        .int()
        .min(0, 'Start time must be between 0 and 1439.')
        .max(1439, 'Start time must be between 0 and 1439.'))
        .optional(),
    endMinutes: zod_1.z
        .string()
        .transform(val => parseInt(val, 10))
        .pipe(zod_1.z.number()
        .int()
        .min(0, 'End time must be between 0 and 1439.')
        .max(1439, 'End time must be between 0 and 1439.'))
        .optional(),
})
    .refine(data => {
    if (data.startMinutes && data.endMinutes) {
        return data.endMinutes > data.startMinutes;
    }
    return true;
}, 'End time must be after start time.');
/**
 * Collection of schemas responsible for validating all student module requests.
 */
exports.studentSchemas = {
    createStudent: zod_1.z.object({
        body: createStudentBodySchema,
    }),
    getStudentById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
    listAdaptedStudents: zod_1.z.object({
        query: listAdaptedStudentsQuery,
    }),
    updateStudent: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: updateStudentBodySchema,
    }),
    deleteStudent: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
    getTeachers: zod_1.z.object({
        query: getTeachersQuerySchema,
    }),
    createClass: class_schema_1.classSchemas.createClass,
    getClasses: class_schema_1.classSchemas.getClasses,
    getClassById: class_schema_1.classSchemas.getClassById,
    updateClass: class_schema_1.classSchemas.updateClass,
};
