"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.teacherSchemas = exports.teacherSchema = void 0;
const zod_1 = require("zod");
const uuidv7_schema_1 = require("./shared/uuidv7.schema");
const email_schema_1 = require("./shared/email.schema");
const password_schema_1 = require("./shared/password.schema");
const simpleString_schema_1 = require("./shared/simpleString.schema");
const cpf_schema_1 = require("./shared/cpf.schema");
const rg_schema_1 = require("./shared/rg.schema");
const telephone_schema_1 = require("./shared/telephone.schema");
const date_schema_1 = require("./shared/date.schema");
const cep_schema_1 = require("./shared/cep.schema");
const class_schema_1 = require("./class.schema");
const cnpj_schema_1 = require("./shared/cnpj.schema");
/**
 * Utility function to validate Brazilian state (UF)
 */
function isValidState(state) {
    const validStates = [
        'AC', 'AL', 'AP', 'AM', 'BA', 'CE', 'DF', 'ES', 'GO', 'MA',
        'MT', 'MS', 'MG', 'PA', 'PB', 'PR', 'PE', 'PI', 'RJ', 'RN',
        'RS', 'RO', 'RR', 'SC', 'SP', 'SE', 'TO'
    ];
    return validStates.includes(state.toUpperCase());
}
/**
 * Utility function to validate payment type and key (PIX, bank account, etc.)
 */
function isValidPaymentInfo(type, key) {
    const validTypes = ['pix', 'bank_account', 'digital_wallet'];
    if (!validTypes.includes(type)) {
        return false;
    }
    // Basic validation for PIX key (can be CPF, email, phone, or random key)
    if (type === 'pix') {
        // PIX keys can be: CPF/CNPJ, email, phone, or random key
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        const phoneRegex = /^\+55\d{10,11}$/;
        const cpfCnpjRegex = /^\d{11,14}$/;
        return emailRegex.test(key) || phoneRegex.test(key) || cpfCnpjRegex.test(key) || key.length >= 10;
    }
    // For bank account, validate it's not empty
    if (type === 'bank_account') {
        return key.length >= 5;
    }
    // For digital wallet, validate it's not empty
    if (type === 'digital_wallet') {
        return key.length >= 5;
    }
    return true;
}
/**
 * Shared fields for teacher-related payloads
 */
exports.teacherSchema = {
    full_name: simpleString_schema_1.stringSchemas.portugueseName,
    cpfCnpj: zod_1.z.object({
        value: zod_1.z
            .union([cpf_schema_1.cpfValidation, cnpj_schema_1.cnpjValidation])
            .refine(() => true, { message: "Invalid CPF/CNPJ format" }),
        upload_id: uuidv7_schema_1.uuidV7Schema.optional(),
        type: zod_1.z.enum(['CPF', 'CNPJ']),
    }),
    rg: zod_1.z.object({
        value: rg_schema_1.rgValidation,
        upload_id: uuidv7_schema_1.uuidV7Schema.optional(),
    }).optional(),
    email: email_schema_1.emailValidation,
    password: password_schema_1.passwordValidation,
    telephone: telephone_schema_1.telephoneValidation,
    birth_date: date_schema_1.dateValidation.isoDateSchema
        .transform((val) => new Date(val)) // Transform to Date object after validation
        .pipe(zod_1.z.date()
        .max(new Date(new Date().setFullYear(new Date().getFullYear() - 18)), 'Teacher must be at least 18 years old.')
        .min(new Date(new Date().setFullYear(new Date().getFullYear() - 100)), 'Teacher cannot be older than 100 years.')),
    image_profile: zod_1.z
        .string()
        .trim()
        .url('Provide a valid URL for profile image.')
        .optional(),
    address: zod_1.z.object({
        street: zod_1.z
            .string()
            .trim()
            .min(1, 'Street cannot be empty.')
            .max(200, 'Street cannot exceed 200 characters.'),
        number: zod_1.z
            .string()
            .trim()
            .min(1, 'Number cannot be empty.')
            .max(10, 'Number cannot exceed 10 characters.'),
        complement: zod_1.z
            .string()
            .trim()
            .max(100, 'Complement cannot exceed 100 characters.')
            .optional(),
        city: zod_1.z
            .string()
            .trim()
            .min(1, 'City cannot be empty.')
            .max(100, 'City cannot exceed 100 characters.'),
        state: zod_1.z
            .string()
            .trim()
            .length(2, 'State must be 2 characters (UF).')
            .refine(isValidState, 'Provide a valid Brazilian state (UF).'),
        zip_code: cep_schema_1.zipCodeValidation,
    }).optional(),
    education: zod_1.z.object({
        level: zod_1.z
            .string()
            .trim()
            .min(1, 'Education level cannot be empty.')
            .max(100, 'Education level cannot exceed 100 characters.'),
        field: zod_1.z
            .string()
            .trim()
            .min(1, 'Field of study cannot be empty.')
            .max(100, 'Field of study cannot exceed 100 characters.'),
        teaching_experience: zod_1.z
            .string()
            .trim()
            .min(1, 'Teaching experience cannot be empty.')
            .max(50, 'Teaching experience cannot exceed 50 characters.'),
        disciplines: uuidv7_schema_1.uuidV7ArraySchema,
        education_levels: zod_1.z
            .array(zod_1.z.string()
            .trim()
            .min(1, 'Education level cannot be empty.')
            .max(50, 'Education level cannot exceed 50 characters.'))
            .min(1, 'Provide at least one education level.')
            .max(10, 'Cannot exceed 10 education levels.'),
    }).optional(),
    availability: zod_1.z.object({
        in_person_location: zod_1.z
            .array(zod_1.z.string()
            .trim()
            .min(1, 'Location cannot be empty.')
            .max(100, 'Location cannot exceed 100 characters.'))
            .min(1, 'Provide at least one in-person location.')
            .max(5, 'Cannot exceed 5 locations.'),
        available_times: zod_1.z
            .array(zod_1.z.object({
            day_of_week: zod_1.z
                .number()
                .int()
                .min(0, 'Day of week must be between 0 and 6.')
                .max(6, 'Day of week must be between 0 and 6.'),
            start_time: zod_1.z
                .number()
                .int()
                .min(0, 'Start time must be between 0 and 1439.')
                .max(1439, 'Start time must be between 0 and 1439.'),
            end_time: zod_1.z
                .number()
                .int()
                .min(0, 'End time must be between 0 and 1439.')
                .max(1439, 'End time must be between 0 and 1439.'),
        }))
            .min(1, 'Provide at least one available time.')
            .refine(times => {
            return times.every(time => time.end_time > time.start_time);
        }, 'End time must be after start time for all available times.'),
        accepts_emergency_classes: zod_1.z.boolean().default(false),
    }).optional(),
    payment: zod_1.z.object({
        hourly_rate: zod_1.z
            .number()
            .positive('Hourly rate must be positive.')
            .min(10, 'Hourly rate must be at least R$ 10.00.')
            .max(1000, 'Hourly rate cannot exceed R$ 1000.00.'),
        payment_info: zod_1.z.object({
            type: zod_1.z.enum(['pix', 'bank_account', 'digital_wallet']),
            key: zod_1.z
                .string()
                .trim()
                .min(1, 'Payment key cannot be empty.')
                .max(200, 'Payment key cannot exceed 200 characters.'),
        }).refine((data) => isValidPaymentInfo(data.type, data.key), {
            message: 'Provide valid payment information for the selected type.',
            path: ['key'],
        }),
    }).optional(),
    preferences: zod_1.z.object({
        student_type_preference: zod_1.z
            .string()
            .trim()
            .max(200, 'Student type preference cannot exceed 200 characters.')
            .optional(),
        general_observations: zod_1.z
            .string()
            .trim()
            .max(500, 'General observations cannot exceed 500 characters.')
            .optional(),
    }).optional(),
};
/**
 * Schema for creating a new teacher
 */
const createTeacherBodySchema = zod_1.z.object({
    full_name: exports.teacherSchema.full_name,
    cpfCnpj: exports.teacherSchema.cpfCnpj,
    rg: exports.teacherSchema.rg,
    email: exports.teacherSchema.email,
    password: exports.teacherSchema.password,
    telephone: exports.teacherSchema.telephone,
    birth_date: exports.teacherSchema.birth_date,
    image_profile: exports.teacherSchema.image_profile,
    address: exports.teacherSchema.address,
    education: exports.teacherSchema.education,
    availability: exports.teacherSchema.availability,
    payment: exports.teacherSchema.payment,
    preferences: exports.teacherSchema.preferences,
});
/**
 * Schema for updating an existing teacher
 */
const updateTeacherBodySchema = zod_1.z
    .object({
    full_name: exports.teacherSchema.full_name.optional(),
    cpfCnpj: exports.teacherSchema.cpfCnpj.optional(),
    rg: exports.teacherSchema.rg.optional(),
    telephone: exports.teacherSchema.telephone.optional(),
    birth_date: exports.teacherSchema.birth_date.optional(),
    image_profile: exports.teacherSchema.image_profile.optional(),
    address: exports.teacherSchema.address.optional(),
    education: exports.teacherSchema.education.optional(),
    availability: exports.teacherSchema.availability.optional(),
    payment: exports.teacherSchema.payment.optional(),
    preferences: exports.teacherSchema.preferences.optional(),
})
    .superRefine((data, ctx) => {
    if (Object.keys(data).length === 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Provide at least one field to update.',
            path: [],
        });
    }
    // Validate that if education is provided, all required fields are present
    if (data.education && Object.keys(data.education).length > 0) {
        const requiredFields = ['level', 'field', 'teaching_experience', 'disciplines', 'education_levels'];
        const missingFields = requiredFields.filter(field => { var _a; return !((_a = data.education) === null || _a === void 0 ? void 0 : _a[field]); });
        if (missingFields.length > 0) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: `When updating education, provide all required fields: ${requiredFields.join(', ')}.`,
                path: ['education'],
            });
        }
    }
    // Validate that if availability is provided, all required fields are present
    if (data.availability && Object.keys(data.availability).length > 0) {
        const requiredFields = ['available_times'];
        const missingFields = requiredFields.filter(field => { var _a; return !((_a = data.availability) === null || _a === void 0 ? void 0 : _a[field]); });
        if (missingFields.length > 0) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: `When updating availability, provide all required fields: ${requiredFields.join(', ')}.`,
                path: ['availability'],
            });
        }
    }
    // Validate that if payment is provided, all required fields are present
    if (data.payment && Object.keys(data.payment).length > 0) {
        if (!data.payment.hourly_rate || !data.payment.payment_info) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: 'When updating payment, provide both hourly_rate and payment_info.',
                path: ['payment'],
            });
        }
    }
});
/**
 * Query parameters for filtering classes to teacher
 */
const listClassesQuerySchema = class_schema_1.getClassesQuerySchema
    .omit({
    student_id: true,
})
    .required({
    teacher_id: true,
});
/**
 * Collection of schemas responsible for validating all teacher module requests.
 */
exports.teacherSchemas = {
    createTeacher: zod_1.z.object({
        body: createTeacherBodySchema,
    }),
    getTeacherById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
    updateTeacher: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: updateTeacherBodySchema,
    }),
    deleteTeacher: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
    getClasses: zod_1.z.object({
        query: listClassesQuerySchema,
    }),
    getClassById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
};
