"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.topicSchemas = exports.topicSchema = void 0;
const zod_1 = require("zod");
const uuidv7_schema_1 = require("./shared/uuidv7.schema");
/**
 * Shared fields for topic-related payloads
 */
exports.topicSchema = {
    name: zod_1.z
        .string()
        .trim()
        .min(1, 'Topic name cannot be empty.')
        .max(100, 'Topic name cannot exceed 100 characters.')
        .regex(/^[a-zA-Z0-9À-ÿ\s\-_&()]+$/, 'Topic name can only contain letters, numbers, spaces, hyphens, underscores, ampersands, and parentheses.')
        .refine((name) => {
        // Check for consecutive special characters
        return !/(\s{2,}|_{2,}|-{2,}|\(\))/.test(name);
    }, {
        message: 'Topic name cannot contain consecutive special characters, spaces, or empty parentheses.',
    }),
    description: zod_1.z
        .string()
        .trim()
        .min(1, 'Topic description cannot be empty.')
        .max(500, 'Topic description cannot exceed 500 characters.')
        .optional(),
    level: zod_1.z
        .enum(['EF I', 'EF II', 'EM', 'ES'], {
        message: 'Level must be one of: EF I, EF II, EM, ES',
    }),
    discipline_id: uuidv7_schema_1.uuidV7Schema,
};
/**
 * Schema for creating a new topic
 */
const createTopicBodySchema = zod_1.z.object({
    name: exports.topicSchema.name,
    description: exports.topicSchema.description,
    level: exports.topicSchema.level,
    discipline_id: exports.topicSchema.discipline_id,
});
/**
 * Query parameters for filtering topics
 */
const listTopicsQuerySchema = zod_1.z.object({
    discipline_id: exports.topicSchema.discipline_id.optional(),
    level: exports.topicSchema.level.optional(),
});
/**
 * Schema for updating an existing topic
 */
const updateTopicBodySchema = zod_1.z
    .object({
    name: exports.topicSchema.name.optional(),
    description: exports.topicSchema.description,
    level: exports.topicSchema.level.optional(),
    discipline_id: exports.topicSchema.discipline_id.optional(),
})
    .superRefine((data, ctx) => {
    if (Object.keys(data).length === 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Provide at least one field to update.',
            path: [],
        });
    }
    // Validate that if description is provided, it's not empty
    if (data.description !== undefined && data.description.trim() === '') {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Description cannot be empty if provided.',
            path: ['description'],
        });
    }
    // Validate that name doesn't contain only special characters when provided
    if (data.name !== undefined) {
        const nameWithoutSpecialChars = data.name.replace(/[^a-zA-Z0-9À-ÿ]/g, '');
        if (nameWithoutSpecialChars.length === 0) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: 'Topic name must contain at least one letter or number.',
                path: ['name'],
            });
        }
    }
    // Validate that discipline_id is a valid UUID v7 if provided
    if (data.discipline_id !== undefined) {
        if (data.discipline_id.trim() === '') {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                message: 'Category ID cannot be empty if provided.',
                path: ['discipline_id'],
            });
        }
    }
});
/**
 * Collection of schemas responsible for validating all topic module requests.
 */
exports.topicSchemas = {
    createTopic: zod_1.z.object({
        body: createTopicBodySchema,
    }),
    getTopic: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
    listTopics: zod_1.z.object({
        query: listTopicsQuerySchema,
    }),
    updateTopic: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: updateTopicBodySchema,
    }),
    deleteTopic: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }),
};
