"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.userSchemas = exports.userRegisterSchema = void 0;
const zod_1 = require("zod");
const uuidv7_schema_1 = require("./shared/uuidv7.schema");
const email_schema_1 = require("./shared/email.schema");
const password_schema_1 = require("./shared/password.schema");
const simpleString_schema_1 = require("./shared/simpleString.schema");
const student_schema_1 = require("./student.schema");
const teacher_schema_1 = require("./teacher.schema");
/**
 * Schema used to validate user registration payloads (matches IParamsCreateUser).
 */
exports.userRegisterSchema = zod_1.z.object({
    name: simpleString_schema_1.stringSchemas.portugueseName.optional(),
    email: email_schema_1.emailValidation,
    password: password_schema_1.passwordValidation,
});
/**
 * Schema that validates a non-empty list of role IDs
 */
const rolesIdsSchema = uuidv7_schema_1.uuidV7ArraySchema.min(1, {
    message: 'Provide at least one role ID',
});
/**
 * Schema describing the payload accepted when updating an existing user.
 */
const updateUserBodySchema = zod_1.z
    .object({
    name: simpleString_schema_1.stringSchemas.portugueseName.optional(),
    is_active: zod_1.z.boolean().optional(),
    roles: zod_1.z.array(uuidv7_schema_1.uuidV7Schema).optional(),
})
    .superRefine((data, ctx) => {
    if (Object.keys(data).length === 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Provide at least one field to update.',
            path: [],
        });
    }
});
/**
 * Collection of schemas responsible for validating all user module requests.
 */
exports.userSchemas = Object.assign(Object.assign({ createUser: zod_1.z.object({
        body: exports.userRegisterSchema.extend({
            name: zod_1.z
                .string()
                .trim()
                .min(1, 'Name cannot be empty.'),
        }),
    }), listUsers: zod_1.z.object({
        query: zod_1.z
            .object({
            name: zod_1.z.string().trim().optional(),
            email: zod_1.z.string().trim().optional(),
            is_active: zod_1.z
                .enum(['true', 'false'])
                .transform(value => value === 'true')
                .optional(),
        })
            .default({}),
    }), getUserById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }), getUserRolesById: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }), assignRoleToUser: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: zod_1.z.object({
            rolesIds: rolesIdsSchema,
        }),
    }), updateUser: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: updateUserBodySchema,
    }), removeRoleFromUser: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
        body: zod_1.z.object({
            rolesIds: rolesIdsSchema,
        }),
    }), deleteUser: zod_1.z.object({
        params: zod_1.z.object({
            id: uuidv7_schema_1.uuidV7Schema,
        }),
    }) }, student_schema_1.studentSchemas), teacher_schema_1.teacherSchemas);
