"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Server = void 0;
const db_service_1 = require("../infrastructure/database/db.service");
const swagger_service_1 = require("../shared/services/swagger.service");
const customError_1 = require("../shared/utils/customError");
const traceability_1 = require("traceability");
const express_rate_limit_1 = __importDefault(require("express-rate-limit"));
const helmet_1 = __importDefault(require("helmet"));
const cors_1 = __importDefault(require("cors"));
const express_1 = __importDefault(require("express"));
require("express-async-errors");
const zod_1 = require("zod");
const winston = __importStar(require("winston"));
class Server {
    constructor(appInit) {
        // Application protection middlewares
        this.protectMiddlewares = [
            traceability_1.ContextAsyncHooks.getExpressMiddlewareTracking(),
            (0, express_rate_limit_1.default)({
                windowMs: 1000, // 1s
                max: 5, // rate limit 5 requests
                message: {
                    message: 'Request rate limit (burst). Please try again later.'
                },
                standardHeaders: false,
                legacyHeaders: false,
            }),
            (0, express_rate_limit_1.default)({
                windowMs: 60 * 1000, // 1m
                max: 50, // rate limit 50 requests
                message: {
                    message: 'Request rate limit (sustained). Please try again later.'
                },
                standardHeaders: true,
                legacyHeaders: false,
            }),
            (0, helmet_1.default)(),
            (0, cors_1.default)(),
            express_1.default.json({ limit: '3mb' }),
            express_1.default.urlencoded({ limit: '3mb', extended: true }),
        ];
        globalThis.AppError = customError_1.AppError;
        // Configure Logger to json format
        const consoleTransport = traceability_1.Logger.transports.find((transport) => transport instanceof winston.transports.Console);
        if (consoleTransport) {
            consoleTransport.format = winston.format.combine(winston.format.timestamp(), winston.format.json({ space: 2 }));
        }
        this.app = (0, express_1.default)();
        this.port = appInit.port;
        this.basePath = appInit.basePath;
        this.routes = appInit.routers || [];
        this.timeoutMilliseconds = appInit.timeoutMilliseconds;
        this.databaseService = new db_service_1.DatabaseService();
        this.swaggerService = new swagger_service_1.SwaggerService(this.app, this.basePath);
    }
    /**
     * Start the server
     */
    start() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                // Database connection
                yield this.databaseService.connect();
                // Middlewares
                this.middlewares(this.protectMiddlewares);
                // Request logger middlewares
                this.requestLogger();
                // Swagger documentation
                this.swaggerService.setupSwaggerWebDocs();
                // Swagger validator router
                this.swaggerService.swaggerValidator();
                // Test server health
                this.app.get(`${this.basePath}/health`, (req, res) => {
                    res.status(200).json({ status: 'OK' });
                });
                // Configure routers
                this.configureRouters(this.routes);
                // Handler not found paths
                this.notFoundHandler();
                // Swagger error manipulator
                this.app.use(this.swaggerService.getErrorHandler());
                // Generic error manipulator
                this.errorHandler();
                // Init application
                this.listen();
            }
            catch (error) {
                traceability_1.Logger.error(`Failed to start application: ${error.message}`);
            }
        });
    }
    /**
     * Configure the application middlewares
     */
    middlewares(middleWares) {
        middleWares.forEach((middleWare) => this.app.use(middleWare));
    }
    /**
     * Error handler
     */
    errorHandler() {
        this.app.use((err, req, res, _next) => {
            // Correctly handles and returns the error
            if (err instanceof zod_1.ZodError) {
                // Validation error with Zod
                res.status(400).json({
                    message: 'Validation failed',
                    issues: err.issues.map(issue => ({
                        path: issue.path.join('.'),
                        message: issue.message,
                    })),
                });
            }
            else if (err instanceof customError_1.AppError && err.status !== 500) {
                // Application error
                res.status(err.status).json({ message: err.message });
            }
            else {
                // Internal server error
                // Only logs generic server errors
                console.log(err);
                res.status(500).json({ message: `Internal server error` });
            }
        });
    }
    /**
     * Request logger
     */
    requestLogger() {
        this.app.use((req, res, next) => {
            // Saves the message sent to the user
            const originalJson = res.json;
            res.json = (body) => {
                if (body && body.message) {
                    res.locals.responseMessage = body.message;
                }
                return originalJson.apply(res, [body]);
            };
            // Logs the request data after sending the response
            res.on('finish', () => {
                const message = res.locals.responseMessage || "";
                traceability_1.Logger.info(message, {
                    method: req.method,
                    path: req.originalUrl,
                    status: res.statusCode,
                    user: req.user || "anonymous",
                    ip: req.ip,
                });
            });
            next();
        });
    }
    /**
     * Handler for undefined routes
     */
    notFoundHandler() {
        this.app.use((
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        req, res, next) => {
            throw new customError_1.AppError('Route not found', 404);
        });
    }
    /**
     * Configure all the routers
     */
    configureRouters(routers) {
        routers.forEach((router) => {
            this.app.use(`${this.basePath}${router.path}`, // Declares the endpoint
            router.getRoutes() // Declare the routes for endpoint
            );
        });
    }
    /**
     * Inicialize the server for listen the requests
     */
    listen() {
        return this.app.listen(this.port, () => {
            traceability_1.Logger.info(`App listening on the http://localhost:${this.port}`);
        }).setTimeout(this.timeoutMilliseconds);
    }
}
exports.Server = Server;
