"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthService = void 0;
const bcrypt_service_1 = require("../../shared/services/bcrypt.service");
const jwt_service_1 = require("../../shared/services/jwt.service");
const mailtrap_service_1 = require("../../shared/services/mailtrap.service");
const codeGenerator_1 = require("../../shared/utils/codeGenerator");
/**
 * Generic authentication service that can work with any authenticatable entity
 * @template T - Type of entity that extends IAuthenticatableEntity
 * @template R - Type of registration parameters that extends IRegisterParams
 */
class AuthService {
    constructor({ repository, entity }) {
        this.repository = repository;
        this.Entity = entity;
        this.bcryptService = new bcrypt_service_1.BcryptService();
        this.jwtService = new jwt_service_1.JwtService();
        this.mailtrapService = new mailtrap_service_1.MailtrapService();
        this.generateCode = codeGenerator_1.generateCode;
    }
    /**
     * Register a new user
     * @param params - The user data to register
     */
    register(params) {
        return __awaiter(this, void 0, void 0, function* () {
            const newEntity = this.Entity(params);
            const existingUser = yield this.repository.findByEmail(params.email);
            if (existingUser)
                throw new AppError('Email already in use', 400);
            newEntity.password = this.bcryptService.hashPassword(params.password);
            return yield this.repository.create(newEntity);
        });
    }
    /**
     * Login a user
     * @param params - The user data to login
     * @returns The JWT token
     */
    login(params) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!params.email || !params.password)
                throw new AppError('Email and password are required', 400);
            const user = yield this.repository.findByEmail(params.email);
            if (!user)
                throw new AppError('Incorrect email or password', 401);
            const isValidPassword = this.bcryptService.validatePassword(params.password, user.password);
            if (!isValidPassword)
                throw new AppError('Incorrect email or password', 401);
            // Generate and return JWT token
            const payload = {
                id: user.id,
                email: user.email,
            };
            return this.jwtService.generateToken(payload);
        });
    }
    getProfile(id) {
        return __awaiter(this, void 0, void 0, function* () {
            const user = yield this.repository.findById(id);
            if (!user)
                throw new AppError('User not found', 404);
            return user;
        });
    }
    updateProfile(params) {
        return __awaiter(this, void 0, void 0, function* () {
            const oldUser = yield this.repository.findById(params.id);
            if (!oldUser)
                throw new AppError('User not found', 404);
            const newData = Object.assign(Object.assign({}, oldUser), params.data);
            const newUser = yield this.repository.update(params.id, newData);
            if (!newUser)
                throw new AppError('User could not be updated', 500);
            return newUser;
        });
    }
    /**
     * Verify a JWT token and return the associated user
     * Used in middleware
     * @param token - The JWT token to verify
     */
    verifyToken(token) {
        return __awaiter(this, void 0, void 0, function* () {
            // Verify the token and extract the payload
            const payload = this.jwtService.verifyToken(token);
            if (!payload)
                return null;
            // Confirm the user still exists
            const user = yield this.repository.findById(payload.id);
            if (!user)
                return null;
            return payload;
        });
    }
    /**
     * Sends the password reset link to the registered email
     * @param email - User email
     */
    forgotPassword(email) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!email)
                throw new AppError('Email are required', 400);
            const user = yield this.repository.findByEmail(email);
            if (!user)
                throw new AppError('User not found', 404);
            // Generate the code and calculate the expires time
            const code = this.generateCode();
            const now = new Date();
            const expiresAt = new Date(now.getTime() + 30 * 60000); // 30m
            const userWithCode = Object.assign(Object.assign({}, user), { password_reset: {
                    token: code,
                    expires_at: expiresAt,
                } });
            // Save token and espires at in database
            const newUseModel = yield this.repository.update(user.id, userWithCode);
            // Send the email
            yield this.mailtrapService.sendOneMail({
                recipient: user.email,
                subject: "Reset your password with the code.",
                text: `Your code is: ${code}`,
            });
            return newUseModel;
        });
    }
    /**
     * Reset user password using verification code
     * @param code - The code sent to the user
     * @param email - The user's email
     * @param newPassword - The new password
     */
    resetPassword(code, email, newPassword) {
        return __awaiter(this, void 0, void 0, function* () {
            var _a, _b;
            // Validate required parameters
            if (!email || !code || !newPassword) {
                throw new AppError('Email, verification code, and new password are required', 400);
            }
            // Verify if the email belongs to an existing user
            const user = yield this.repository.findByEmail(email);
            if (!user) {
                throw new AppError('Invalid or expired password reset token', 401);
            }
            // Validate code is correct
            if (((_a = user.password_reset) === null || _a === void 0 ? void 0 : _a.token) != code) {
                throw new AppError('Invalid or expired password reset token', 401);
            }
            // Verify if code not expires
            const now = new Date();
            if (now > ((_b = user.password_reset) === null || _b === void 0 ? void 0 : _b.expires_at)) {
                throw new AppError('Invalid or expired password reset token', 401);
            }
            // Hash new password
            const hashedPassword = this.bcryptService.hashPassword(newPassword);
            // Save in database
            return yield this.repository.update(user.id, Object.assign(Object.assign({}, user), { password: hashedPassword, password_reset: null }));
        });
    }
}
exports.AuthService = AuthService;
