"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RoleService = void 0;
const roles_entity_1 = require("./roles.entity");
/**
 * Service responsible for business logic related to Roles.
 * Handles creation, retrieval, update, and deletion of roles.
 */
class RoleService {
    constructor({ RoleRepository }) {
        /**
         * Create a new role
         * @param params - Role data to create
         * @returns The created role object
         */
        this.createRole = (params) => __awaiter(this, void 0, void 0, function* () {
            const role = new roles_entity_1.Role(params);
            const exists = yield this.repository.findRoleByName(params.name);
            if (exists)
                throw new AppError('Name already in use', 400);
            return this.repository.createRole(role);
        });
        /**
         * Find a role by its ID
         * @param id - Role ID
         * @returns The role object or null if not found
         */
        this.findRoleById = (id) => __awaiter(this, void 0, void 0, function* () {
            const role = yield this.repository.findRoleById(id);
            if (!role)
                throw new AppError('Role not found', 404);
            return role;
        });
        /**
         * List all roles with optional filter
         * @param filter - Filter object
         * @returns Array of roles
         */
        this.listRoles = (...args_1) => __awaiter(this, [...args_1], void 0, function* (filter = {}) {
            return this.repository.listRole(filter);
        });
        /**
         * Update an existing role
         * @param params - Update parameters including role ID and new data
         * @returns The updated role object or null if not found
         */
        this.updateRole = (params) => __awaiter(this, void 0, void 0, function* () {
            const role = yield this.repository.findRoleById(params.id);
            if (!role)
                throw new AppError('Role not found', 404);
            const newRole = new roles_entity_1.Role(Object.assign(Object.assign({}, role), params.data));
            if (params.data.name && params.data.name !== newRole.name) {
                const existing = yield this.repository.findRoleByName(params.data.name);
                if (existing)
                    throw new AppError('Name already in use', 400);
            }
            return yield this.repository.updateRoleById(params.id, newRole);
        });
        /**
         * Delete a role by its ID
         * @param id - Role ID
         * @returns The deleted role object or null if not found
         */
        this.deleteRole = (id) => __awaiter(this, void 0, void 0, function* () {
            const role = yield this.repository.findRoleById(id);
            if (!role)
                throw new AppError('Role not found', 404);
            return yield this.repository.deleteRoleById(id);
        });
        this.repository = RoleRepository;
    }
}
exports.RoleService = RoleService;
