"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Student = void 0;
class Student {
    constructor(student) {
        var _a, _b;
        // Validate required fields
        Student.validateFields(student);
        // Validate guardians for underage students
        Student.validateIfUnderage(student);
        // Validate format fields
        Student.validateFormatFields(student);
        // Validate the ranking and the total classes
        student.ranking = (_a = student.ranking) !== null && _a !== void 0 ? _a : 0;
        student.total_classes = (_b = student.total_classes) !== null && _b !== void 0 ? _b : 0;
        // Set the properties
        this.data = student;
    }
    getData() {
        return this.data;
    }
    static validateFormatFields(student) {
        // Mapping regexes
        const regexes = {
            emailRegex: /^[^\s@]+@[^\s@]+\.[^\s@]+$/,
            telephoneRegex: /^\(?\d{2}\)?[\s-]?\d{4,5}-?\d{4}$/,
            cpfRegex: /(^\d{3}\.\d{3}\.\d{3}-\d{2}$)|(^\d{11}$)/,
            rgRegex: /(^\d{1,2}).?(\d{3}).?(\d{3})-?(\d{1}|X|x$)/,
        };
        // Student fields
        if (!regexes['emailRegex'].test(student.email)) {
            throw new AppError('Invalid student email format', 400);
        }
        if (!regexes['telephoneRegex'].test(student.telephone)) {
            throw new AppError('Invalid student telephone format', 400);
        }
        if (!regexes['cpfRegex'].test(student.cpf.value)) {
            throw new AppError('Invalid student CPF format', 400);
        }
        if (student.rg && student.rg.value && !regexes['rgRegex'].test(student.rg.value)) {
            throw new AppError('Invalid student RG format', 400);
        }
        if (!Student.validateBirthDate(student.birth_date)) {
            throw new AppError('Invalid student birth date format', 400);
        }
        // Guardian fields
        if (student.guardian) {
            if (!regexes['emailRegex'].test(student.guardian.email)) {
                throw new AppError('Invalid guardian email format', 400);
            }
            if (!regexes['telephoneRegex'].test(student.guardian.telephone)) {
                throw new AppError('Invalid guardian telephone format', 400);
            }
            if (!regexes['cpfRegex'].test(student.guardian.cpf.value)) {
                throw new AppError('Invalid guardian CPF format', 400);
            }
            if (!regexes['rgRegex'].test(student.guardian.rg.value)) {
                throw new AppError('Invalid guardian RG format', 400);
            }
            if (!Student.validateBirthDate(student.guardian.birth_date)) {
                throw new AppError('Invalid guardian birth date format', 400);
            }
        }
    }
    static validateFields(student) {
        const requiredTopLevel = [
            'full_name',
            'birth_date',
            'email',
            'password',
            'telephone',
            'cpf.value',
            'academic.current_grade',
            'academic.subjects',
            'academic.difficulty_level',
            'academic.main_goal',
            'class_preferences.modality',
            'class_preferences.frequency',
            'class_preferences.available_times'
        ];
        for (const path of requiredTopLevel) {
            const value = Student.getNestedValue(student, path);
            if (!value) {
                throw new AppError(`Missing required field: ${path}`, 400);
            }
            if (Array.isArray(value) && value.length === 0) {
                throw new AppError(`Field cannot be an empty array: ${path}`, 400);
            }
        }
    }
    static getNestedValue(obj, path) {
        return path.split('.').reduce((acc, part) => {
            if (!acc || typeof acc !== 'object') {
                return undefined;
            }
            return acc[part];
        }, obj);
    }
    static validateIfUnderage(student) {
        const today = new Date();
        const birthDateStudent = new Date(student.birth_date);
        // Validate if date is valid
        if (isNaN(birthDateStudent.getTime())) {
            throw new AppError('Invalid student birth date', 400);
        }
        let ageStudent = today.getFullYear() - birthDateStudent.getFullYear();
        const currentMonth = today.getMonth();
        const currentDay = today.getDate();
        const studentBirthMonth = birthDateStudent.getMonth();
        const studentBirthDay = birthDateStudent.getDate();
        if (currentMonth < studentBirthMonth ||
            (currentMonth === studentBirthMonth && currentDay < studentBirthDay)) {
            ageStudent--;
        }
        if (ageStudent < 18) {
            // Check if guardian data is provided
            if (!student.guardian ||
                !student.guardian.cpf.value ||
                !student.guardian.email ||
                !student.guardian.full_name ||
                !student.guardian.relationship ||
                !student.guardian.rg.value ||
                !student.guardian.telephone ||
                !student.guardian.birth_date) {
                throw new AppError(`Guardian's data are required for underage students`, 400);
            }
            const birthDateGuardian = new Date(student.guardian.birth_date);
            if (isNaN(birthDateGuardian.getTime())) {
                throw new AppError('Invalid guardian birth date', 400);
            }
            // Check if guardian is not underage
            let ageGuardian = today.getFullYear() - birthDateGuardian.getFullYear();
            const guardianBirthMonth = birthDateGuardian.getMonth();
            const guardianBirthDay = birthDateGuardian.getDate();
            if (currentMonth < guardianBirthMonth ||
                (currentMonth === guardianBirthMonth && currentDay < guardianBirthDay)) {
                ageGuardian--;
            }
            if (ageGuardian < 18) {
                throw new AppError('Guardian must be at least 18 years old', 400);
            }
        }
    }
    static validateBirthDate(date) {
        const dateString = date instanceof Date ? date.toISOString() : date;
        const isoRegex = /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d{3}Z$/;
        if (!isoRegex.test(dateString)) {
            return false;
        }
        const validDate = new Date(dateString);
        if (isNaN(validDate.getTime())) {
            return false;
        }
        return validDate.toISOString() === dateString;
    }
}
exports.Student = Student;
