"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StudentService = void 0;
const bcrypt_service_1 = require("../../shared/services/bcrypt.service");
const student_entity_1 = require("./student.entity");
/**
 * Service responsible for business logic related to Teachers.
 * Handles creation, retrieval, update, and deletion of teachers.
 */
class StudentService {
    constructor({ repository }) {
        /**
         * Creates a new student
         * @param student - Data for student creation
         * @returns The created student
         */
        this.createStudent = (params) => __awaiter(this, void 0, void 0, function* () {
            // Create a new instance os Student
            const newStudent = new student_entity_1.Student(Object.assign(Object.assign({}, params), { ranking: 0, total_classes: 0 }));
            // Verify if exist
            const studentExisting = yield this.repository.getStudentByEmail(params.email);
            if (studentExisting)
                throw new AppError('Email already in use', 400);
            // Get the data
            const studentData = newStudent.getData();
            // Hash the password before save in database
            studentData.password = this.bcryptService.hashPassword(params.password);
            return yield this.repository.createStudent(studentData);
        });
        /**
         * Finds a student by ID
         * @param id - Student ID
         * @returns The found student or throws error if not found
         */
        this.getStudentById = (id) => __awaiter(this, void 0, void 0, function* () {
            const student = yield this.repository.getStudentById(id);
            if (!student)
                throw new AppError('Student not found', 404);
            return student;
        });
        /**
         * Lists students according to filter
         * @param filter - Search filter
         * @returns Array of found students
         */
        this.listStudents = (filter) => __awaiter(this, void 0, void 0, function* () {
            return yield this.repository.listStudents(filter || {});
        });
        /**
         * List students for other students
         * We only search for students with the correct ranking and compatible availability
         * @param filter - Specific search filter
         * @returns Array of found students
         */
        this.listAdaptedStudents = (filter) => __awaiter(this, void 0, void 0, function* () {
            const query = this.buildAvailabilityPipeline(filter);
            const allStudents = yield this.repository.listStudents(query);
            const permittedStudents = this.validateRankingOfStudents(allStudents);
            return this.adapterStudent(permittedStudents);
        });
        /**
         * Updates a student by ID
         * @param params - Update parameters (id and data)
         * @returns The updated student
         */
        this.updateStudent = (params) => __awaiter(this, void 0, void 0, function* () {
            const oldStudent = yield this.repository.getStudentById(params.id);
            if (!oldStudent)
                throw new AppError('Student not found', 404);
            // Performs the merge
            const mergedStudent = this.deepMerge(oldStudent, params);
            // Create new instance of Student
            const student = new student_entity_1.Student(mergedStudent);
            // Get the student data
            const studentData = student.getData();
            if (params.data.password) {
                studentData.password = this.bcryptService.hashPassword(params.data.password);
            }
            return yield this.repository.updateStudent(params.id, studentData);
        });
        /**
         * Deletes a student by ID
         * @param id - Student ID
         * @returns The deleted student
         */
        this.deleteStudent = (id) => __awaiter(this, void 0, void 0, function* () {
            const student = yield this.repository.getStudentById(id);
            if (!student)
                throw new AppError('Student not found', 404);
            return yield this.repository.deleteStudent(id);
        });
        /**
         * Performs deep merging between two documents
         * @param target - Base object to merge into
         * @param source - Object containing new values to merge
         * @returns Merged object
         */
        this.deepMerge = (target, source) => {
            if (!source)
                return target;
            const isObject = (obj) => typeof obj === "object" && obj !== null && !Array.isArray(obj);
            const isArray = (arr) => Array.isArray(arr);
            for (const key of Object.keys(source)) {
                const sourceValue = source[key];
                const targetValue = target[key];
                if (isArray(sourceValue)) {
                    const mergedArray = isArray(targetValue)
                        ? [...targetValue, ...sourceValue]
                        : [...sourceValue];
                    const uniqueArray = mergedArray.filter((item, index, self) => index ===
                        self.findIndex((t) => isObject(t) && isObject(item)
                            ? JSON.stringify(t) === JSON.stringify(item)
                            : t === item));
                    target[key] = uniqueArray;
                }
                else if (isObject(sourceValue)) {
                    const mergedObject = this.deepMerge(isObject(targetValue)
                        ? Object.assign({}, targetValue) : {}, sourceValue);
                    target[key] = mergedObject;
                }
                else if (sourceValue !== undefined) {
                    target[key] = sourceValue;
                }
            }
            return target;
        };
        /**
         * Adapter for sending data from student.
         * @param students - Array of students document
         * @returns Adapted array of students
         */
        this.adapterStudent = (students) => {
            return students.map(student => {
                return {
                    name: student.full_name,
                    cpf: student.cpf.value,
                    birth_date: student.birth_date,
                    available_times: student.class_preferences.available_times,
                };
            });
        };
        /**
         * Validates if a student is permitted to schedule a group class.
         * @param student - The student's data.
         * @returns - Array of students
         */
        this.validateRankingOfStudents = (students) => {
            return students.filter(student => student.ranking >= 3 &&
                student.total_classes >= 5);
        };
        /**
         * Constructs MongoDB aggregation pipeline for availability filtering
         * @param filter - Filter criteria for building the pipeline
         * @returns Array of pipeline stages
         */
        this.buildAvailabilityPipeline = (filter) => {
            const pipeline = [];
            const matchConditions = {};
            // Build Availability Condition
            const availabilityCondition = {};
            // Filter by days of week
            if (filter.daysOfWeek && filter.daysOfWeek.length > 0) {
                availabilityCondition.day_of_week = { $in: filter.daysOfWeek };
            }
            // Filter by start time
            if (Number.isInteger(filter.startMinutes)) {
                availabilityCondition.start_time = { $lte: filter.startMinutes };
            }
            // Filter by end time
            if (Number.isInteger(filter.endMinutes)) {
                availabilityCondition.end_time = { $gte: filter.endMinutes };
            }
            // Apply Availability Condition to $elemMatch
            if (Object.keys(availabilityCondition).length > 0) {
                matchConditions["class_preferences.available_times"] = {
                    $elemMatch: availabilityCondition
                };
            }
            // Apply pipeline only if *any* filters exist (discipline or availability)
            if (Object.keys(matchConditions).length > 0) {
                pipeline.push({ $match: matchConditions });
            }
            return pipeline;
        };
        /**
         * Retrieves teachers based on specified filters
         * @param filter - Criteria for filtering teachers
         * @returns List of teachers matching the filter criteria
         */
        this.getTeachers = (filter) => __awaiter(this, void 0, void 0, function* () {
            return yield this.listTeachersDep(filter);
        });
        this.createClass = (params) => __awaiter(this, void 0, void 0, function* () {
            return yield this.createClassDep(params);
        });
        this.getClassById = (id) => __awaiter(this, void 0, void 0, function* () {
            return yield this.getClassByIdDep(id);
        });
        this.listClasses = (filter) => __awaiter(this, void 0, void 0, function* () {
            return yield this.listClassesDep(filter);
        });
        this.updateClass = (paramas) => __awaiter(this, void 0, void 0, function* () {
            return yield this.updateClassDep(paramas);
        });
        this.cancelClass = (id) => __awaiter(this, void 0, void 0, function* () {
            return yield this.updateClassDep({ id,
                data: {
                    status: "canceled"
                }
            });
        });
        this.repository = repository;
        this.bcryptService = new bcrypt_service_1.BcryptService();
    }
}
exports.StudentService = StudentService;
