"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Teacher = void 0;
class Teacher {
    constructor(teacher) {
        // Validate required fields
        Teacher.validateRequiredFields(teacher);
        // Validate if teacher is underage
        Teacher.validateIfUnderage(teacher);
        // Validate format fields
        Teacher.validateFormatFields(teacher);
        // Validate birth date format
        Teacher.validateBirthDate(teacher.birth_date);
        // Validate avaliable times
        Teacher.validateAvaliableTimes(teacher);
        // Required fields
        this.data = teacher;
    }
    ;
    /**
     * Return the teacher's data because is private
     * @returns the teacher's data
     */
    getData() {
        return this.data;
    }
    static validateFormatFields(teacher) {
        // Mapping regexes
        const regexPatterns = {
            emailRegex: /^[^\s@]+@[^\s@]+\.[^\s@]+$/,
            telephoneRegex: /^\(?\d{2}\)?[\s-]?\d{4,5}-?\d{4}$/,
            cpfRegex: /(^\d{3}\.\d{3}\.\d{3}-\d{2}$)|(^\d{11}$)/,
            rgRegex: /(^\d{1,2}).?(\d{3}).?(\d{3})-?(\d{1}|X|x$)/,
        };
        if (!regexPatterns['emailRegex'].test(teacher.email)) {
            throw new AppError('Invalid teacher email format', 400);
        }
        if (!regexPatterns['telephoneRegex'].test(teacher.telephone)) {
            throw new AppError('Invalid teacher telephone format', 400);
        }
        if (teacher.rg && !regexPatterns['rgRegex'].test(teacher.rg.value)) {
            throw new AppError('Invalid teacher RG format', 400);
        }
        if (teacher.cpfCnpj.type === 'CPF') {
            if (!regexPatterns['cpfRegex'].test(teacher.cpfCnpj.value)) {
                throw new AppError('Invalid teacher CPF format', 400);
            }
        }
        else {
            if (!Teacher.validateCNPJ(teacher.cpfCnpj.value)) {
                throw new AppError('Invalid teacher CNPJ format', 400);
            }
        }
    }
    /**
     * Validate the required fields to create a new Teacher instance
     * @param teacher - params to create a new Teacher
     */
    static validateRequiredFields(teacher) {
        const requiredTopLevel = [
            'email',
            'password',
            'full_name',
            'birth_date',
            'telephone',
            'cpfCnpj.value',
            'cpfCnpj.type',
        ];
        for (const path of requiredTopLevel) {
            const value = Teacher.getNestedValue(teacher, path);
            if (!value) {
                throw new AppError(`Missing required field: ${path}`, 400);
            }
            if (Array.isArray(value) && value.length === 0) {
                throw new AppError(`Field cannot be an empty array: ${path}`, 400);
            }
        }
    }
    static getNestedValue(obj, path) {
        return path.split('.').reduce((acc, part) => {
            if (!acc || typeof acc !== 'object') {
                return undefined;
            }
            return acc[part];
        }, obj);
    }
    static validateIfUnderage(teacher) {
        const today = new Date();
        const currentMonth = today.getMonth();
        const currentDay = today.getDate();
        const teacherBirthDate = new Date(teacher.birth_date + 'T00:00:00');
        let teacherAge = today.getFullYear() - teacherBirthDate.getFullYear();
        const teacherBirthMonth = teacherBirthDate.getMonth();
        const teacherBirthDay = teacherBirthDate.getDate();
        if (currentMonth < teacherBirthMonth || (currentMonth === teacherBirthMonth && currentDay < teacherBirthDay)) {
            teacherAge--;
        }
        if (teacherAge < 18) {
            throw new AppError('Teacher must be at least 18 years old', 400);
        }
    }
    static validateBirthDate(date) {
        date = new Date(date).toISOString();
        const isoRegex = /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d{3}Z$/;
        if (!isoRegex.test(date))
            throw new AppError('Invalid teacher birth date format', 400);
        const validDate = new Date(date);
        if (isNaN(validDate.getTime()))
            throw new AppError('Invalid teacher birth date format', 400);
        if (validDate.toISOString() !== date)
            throw new AppError('Invalid teacher birth date format', 400);
    }
    /**
     * Validates the teacher's available time slots.
     *
     * Each day of the week is represented by an integer from 0 (Sunday) to 6 (Saturday).
     * Each time value (start_time, end_time) represents minutes since midnight (h * 60).
     *
     * Validation rules:
     * - `day_of_week` must be between 0 and 6.
     * - `start_time` must be between 0 and 1380 (23:00 in minutes).
     * - `end_time` must be between 60 and 1440 (24:00 in minutes).
     * - `end_time` must be greater than `start_time`.
     *
     * Throws an AppError if any rule is violated.
     *
     * @param {ITeacher} teacher - The teacher object containing availability data.
     * @throws {AppError} If any of the available time entries are invalid.
     */
    static validateAvaliableTimes(teacher) {
        var _a;
        const times = (_a = teacher.availability) === null || _a === void 0 ? void 0 : _a.available_times;
        if (!times || times.length === 0)
            return;
        for (const time of times) {
            // Turn strings in numbers
            const day_of_week = Number(time.day_of_week);
            const start_time = Number(time.start_time);
            const end_time = Number(time.end_time);
            // Validate if numbers is integers
            if (!Number.isInteger(day_of_week) ||
                !Number.isInteger(start_time) ||
                !Number.isInteger(end_time)) {
                throw new AppError('Day of week and time values must be integers', 400);
            }
            // Validate if the day of week is valid
            if (day_of_week < 0 || day_of_week > 6) {
                throw new AppError('Invalid day of week in available times', 400);
            }
            // Validate if the start time is valid
            if (start_time < 0 || start_time > 1380) {
                throw new AppError('Invalid start time in available times', 400);
            }
            // Validate if the end time is valid
            if (end_time < 60 || end_time > 1440) {
                throw new AppError('Invalid end time in available times', 400);
            }
            // Validate if the end time is grater/later than the start time
            if (end_time <= start_time) {
                throw new AppError('End time must be greater than start time', 400);
            }
        }
    }
    static validateCNPJ(cnpj) {
        if (!cnpj)
            return false;
        const cleaned = cnpj.replace(/[^\d]+/g, '');
        if (cleaned.length !== 14)
            return false;
        if (/^(\d)\1+$/.test(cleaned))
            return false;
        const calcDV = (cnpjArr, length) => {
            const weights = length === 12
                ? [5, 4, 3, 2, 9, 8, 7, 6, 5, 4, 3, 2]
                : [6, 5, 4, 3, 2, 9, 8, 7, 6, 5, 4, 3, 2];
            const sum = weights.reduce((acc, weight, i) => acc + cnpjArr[i] * weight, 0);
            const rest = sum % 11;
            return rest < 2 ? 0 : 11 - rest;
        };
        const cnpjDigits = cleaned.split('').map(Number);
        const dv1 = calcDV(cnpjDigits, 12);
        const dv2 = calcDV(cnpjDigits, 13);
        return dv1 === cnpjDigits[12] && dv2 === cnpjDigits[13];
    }
}
exports.Teacher = Teacher;
