"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TeacherService = void 0;
const teacher_entity_1 = require("./teacher.entity");
const bcrypt_service_1 = require("../../shared/services/bcrypt.service");
/**
 * Service responsible for business logic related to Teachers.
 * Handles creation, retrieval, update, and deletion of teachers.
 */
class TeacherService {
    constructor({ repository }) {
        /**
         * Create a new teacher
         * @param params - Teacher data to create
         * @returns The created teacher object
         */
        this.createTeacher = (params) => __awaiter(this, void 0, void 0, function* () {
            const teacher = new teacher_entity_1.Teacher(params);
            const exists = yield this.repository.getTeacherByEmail(params.email);
            if (exists)
                throw new AppError('Email already in use', 400);
            const teacherData = teacher.getData();
            // Validate the topics IDs
            if (teacherData.education && teacherData.education.disciplines.length > 0) {
                teacherData.education.disciplines = yield this.validateTopicsIds(teacherData.education.disciplines);
            }
            // Hash the password
            teacherData.password = this.bcryptService.hashPassword(params.password);
            return yield this.repository.createTeacher(teacherData);
        });
        /**
         * Find a teacher by its ID
         * @param id - Teacher ID
         * @returns The teacher object or null if not found
         */
        this.getTeacherById = (id) => __awaiter(this, void 0, void 0, function* () {
            const teacher = yield this.repository.getTeacherById(id);
            if (!teacher)
                throw new AppError('Teacher not found', 401);
            return teacher;
        });
        /**
         * List all teachers with optional filter
         * @param filter - Filter object
         * @returns Array of teachers
         */
        this.listTeacher = (...args_1) => __awaiter(this, [...args_1], void 0, function* (filter = {}) {
            return yield this.repository.listTeachers(filter);
        });
        /**
         * Update an existing teacher
         * @param params - Update parameters including teacher ID and new data
         * @returns The updated teacher object or null if not found
         */
        this.updateTeacherById = (params) => __awaiter(this, void 0, void 0, function* () {
            const oldTeacher = yield this.repository.getTeacherById(params.id);
            if (!oldTeacher)
                throw new AppError('Teacher not found', 401);
            // Remove null data
            const data = Object.fromEntries(Object.entries(params.data).filter(([_, value]) => value != null));
            if (Object.keys(data).length === 0)
                throw new AppError('Data not provided', 400);
            const teacherData = this.deepMerge(oldTeacher, data);
            // Create new instance with old data, and overlaying new data
            const newTeacher = new teacher_entity_1.Teacher(teacherData);
            const teacher = newTeacher.getData();
            if ((params.data.education && params.data.education.disciplines.length > 0) &&
                (teacher.education && teacher.education.disciplines.length > 0)) {
                teacher.education.disciplines = yield this.validateTopicsIds(teacher.education.disciplines);
            }
            // Hash the password
            if (params.data.password) {
                teacher.password = this.bcryptService.hashPassword(teacher.password);
            }
            return yield this.repository.updateTeacherById(params.id, teacher);
        });
        /**
         * Delete a teacher by its ID
         * @param id - Teacher ID
         * @returns The deleted teacher object or null if not found
         */
        this.deleteTeacherById = (id) => __awaiter(this, void 0, void 0, function* () {
            const teacher = yield this.repository.getTeacherById(id);
            if (!teacher)
                throw new AppError('Teacher not found', 401);
            return yield this.repository.deleteTeacherById(id);
        });
        /**
         * Validates topic IDs before assigning to teacher
         * @param topics - Array of topic IDs to validate
         * @returns Array of valid topic IDs
         * @throws AppError when no valid topics are found
         */
        this.validateTopicsIds = (topics) => __awaiter(this, void 0, void 0, function* () {
            const existingTopics = yield this.listTopicsDep({
                _id: { $in: topics }
            });
            const validIds = existingTopics.map(topic => topic.id);
            if (validIds.length === 0) {
                throw new AppError('Invalid topic IDs', 400);
            }
            return validIds.filter(Boolean);
        });
        /**
         * Performs deep merging between two documents
         * @param target - Base object to merge into
         * @param source - Object containing new values to merge
         * @returns Merged object
         */
        this.deepMerge = (target, source) => {
            if (!source)
                return target;
            const isObject = (obj) => typeof obj === "object" && obj !== null && !Array.isArray(obj);
            const isArray = (arr) => Array.isArray(arr);
            for (const key of Object.keys(source)) {
                const sourceValue = source[key];
                const targetValue = target[key];
                if (isArray(sourceValue)) {
                    const mergedArray = isArray(targetValue)
                        ? [...targetValue, ...sourceValue]
                        : [...sourceValue];
                    const uniqueArray = mergedArray.filter((item, index, self) => index ===
                        self.findIndex((t) => isObject(t) && isObject(item)
                            ? JSON.stringify(t) === JSON.stringify(item)
                            : t === item));
                    target[key] = uniqueArray;
                }
                else if (isObject(sourceValue)) {
                    const mergedObject = this.deepMerge(isObject(targetValue)
                        ? Object.assign({}, targetValue) : {}, sourceValue);
                    target[key] = mergedObject;
                }
                else if (sourceValue !== undefined) {
                    target[key] = sourceValue;
                }
            }
            return target;
        };
        /**
         * Retrieves teachers based on specified filters
         * @param filter - Criteria for filtering teachers
         * @returns List of teachers matching the filter criteria
         */
        this.findTeacherForStudent = (filter) => __awaiter(this, void 0, void 0, function* () {
            // Validate filter parameters
            yield this.validateFilter(filter);
            // Construct aggregation pipeline based on filter
            const pipeline = this.buildAvailabilityPipeline(filter);
            if (pipeline.length === 0) {
                // If filters are not passed
                const teachers = yield this.repository.listTeachers({});
                return this.adapterTeacherForStudent(teachers);
            }
            else {
                // If passed
                const teachers = yield this.repository.findTeachersWithAggregate(pipeline);
                return this.adapterTeacherForStudent(teachers);
            }
        });
        /**
         * Constructs MongoDB aggregation pipeline for availability filtering
         * @param filter - Filter criteria for building the pipeline
         * @returns Array of pipeline stages
         */
        this.buildAvailabilityPipeline = (filter) => {
            const pipeline = [];
            const matchConditions = {};
            // Filter by discipline
            if (typeof filter.disciplineId === 'string' &&
                filter.disciplineId !== '' &&
                filter.disciplineId !== "undefined") {
                matchConditions["education.disciplines"] = filter.disciplineId;
            }
            // Build Availability Condition
            const availabilityCondition = {};
            // Filter by days of week
            if (filter.daysOfWeek && filter.daysOfWeek.length > 0) {
                availabilityCondition.day_of_week = { $in: filter.daysOfWeek };
            }
            // Filter by start time
            if (Number.isInteger(filter.startMinutes)) {
                availabilityCondition.start_time = { $lte: filter.startMinutes };
            }
            // Filter by end time
            if (Number.isInteger(filter.endMinutes)) {
                availabilityCondition.end_time = { $gte: filter.endMinutes };
            }
            // Apply Availability Condition to $elemMatch
            if (Object.keys(availabilityCondition).length > 0) {
                matchConditions["avaliability.available_times"] = {
                    $elemMatch: availabilityCondition
                };
            }
            // Apply pipeline only if *any* filters exist (discipline or availability)
            if (Object.keys(matchConditions).length > 0) {
                pipeline.push({ $match: matchConditions });
            }
            return pipeline;
        };
        /**
         * Validates filter parameters for teacher search
         * @param filter - Filter object to validate
         * @throws AppError when validation fails
         */
        this.validateFilter = (filter) => __awaiter(this, void 0, void 0, function* () {
            if (filter.daysOfWeek !== undefined && filter.daysOfWeek !== null) {
                // Convert to array if single value
                if (!Array.isArray(filter.daysOfWeek)) {
                    filter.daysOfWeek = [filter.daysOfWeek];
                }
                // Convert string values to numbers
                filter.daysOfWeek = filter.daysOfWeek.map(day => Number(day));
                // Validate day of week values
                for (const day of filter.daysOfWeek) {
                    if (day < 0 ||
                        day > 6 ||
                        isNaN(day)) {
                        throw new AppError('Invalid week days', 400);
                    }
                }
            }
            let startTime = undefined;
            let endTime = undefined;
            // Validate the start time
            if (Number.isInteger(filter.startMinutes)) {
                startTime = Number(filter.startMinutes);
                if (!Number.isInteger(startTime)) {
                    throw new AppError('Invalid start time, must be an integer', 400);
                }
                filter.startMinutes = startTime;
            }
            else {
                filter.startMinutes = undefined;
            }
            // Validate the end time
            if (Number.isInteger(filter.endMinutes)) {
                endTime = Number(filter.endMinutes);
                if (!Number.isInteger(endTime)) {
                    throw new AppError('Invalid end time, must be an integer', 400);
                }
                filter.endMinutes = endTime;
            }
            else {
                filter.endMinutes = undefined;
            }
            if (startTime && endTime) {
                if (startTime >= endTime) {
                    throw new AppError('Invalid time range: start time must be less than end time', 400);
                }
            }
            // Validate discipline ID
            if (typeof filter.disciplineId === 'string' &&
                filter.disciplineId !== "undefined" &&
                filter.disciplineId !== "") {
                yield this.validateTopicsIds([filter.disciplineId]);
            }
        });
        /**
         * Adapter for sending data from teacher to student.
         * @param teachers - Array of teacher document
         * @returns Adapted array of teacher
         */
        this.adapterTeacherForStudent = (teachers) => __awaiter(this, void 0, void 0, function* () {
            return yield Promise.all(teachers.map((teacher) => __awaiter(this, void 0, void 0, function* () {
                var _a, _b, _c, _d, _e, _f;
                return {
                    id: teacher.id,
                    full_name: teacher.full_name,
                    email: teacher.email,
                    telephone: teacher.telephone,
                    image_profile: teacher.image_profile,
                    disciplines: (_b = (_a = teacher.education) === null || _a === void 0 ? void 0 : _a.disciplines) !== null && _b !== void 0 ? _b : [],
                    level_education: (_d = (_c = teacher.education) === null || _c === void 0 ? void 0 : _c.level) !== null && _d !== void 0 ? _d : "",
                    available_times: (_f = (_e = teacher.availability) === null || _e === void 0 ? void 0 : _e.available_times) !== null && _f !== void 0 ? _f : [],
                    classes: yield this.listClasses({
                        status: 'scheduled',
                        teacher_id: teacher.id,
                    }),
                };
            })));
        });
        this.listClasses = (filter) => __awaiter(this, void 0, void 0, function* () {
            return yield this.listClassesDep(filter);
        });
        this.getClassById = (id) => __awaiter(this, void 0, void 0, function* () {
            return yield this.getClassByIdDep(id);
        });
        this.finishClass = (id) => __awaiter(this, void 0, void 0, function* () {
            return yield this.updateClassByIdDep({
                id,
                data: {
                    status: "completed"
                },
            });
        });
        this.repository = repository;
        this.bcryptService = new bcrypt_service_1.BcryptService();
    }
}
exports.TeacherService = TeacherService;
