"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.User = void 0;
class User {
    constructor(user) {
        User.validateRequiredFields(user);
        // Validate email format
        if (!/\S+@\S+\.\S+/.test(user.email)) {
            throw new AppError('Invalid email format', 400);
        }
        // Validate password strength
        if (!this.validatePasswordStrength(user.password)) {
            throw new AppError('Password must be at least 8 characters long, include uppercase, lowercase, number, and special character', 400);
        }
        this.data = user;
    }
    getData() {
        return this.data;
    }
    /**
     * Validate the strength of a password
     * @param password - password to validate
     * @returns True if the password is strong, false otherwise
     */
    validatePasswordStrength(password) {
        // Password must be at least 8 characters long
        const minLength = 8;
        // Password must include uppercase
        const hasUpperCase = /[A-Z]/.test(password);
        // Password must include lowercase
        const hasLowerCase = /[a-z]/.test(password);
        // Password must include number
        const hasNumber = /\d/.test(password);
        // Password must include special character
        const hasSpecialChar = /[!@#$%^&*(),.?":{}|<>]/.test(password);
        return password.length >= minLength
            && hasUpperCase
            && hasLowerCase
            && hasNumber
            && hasSpecialChar;
    }
    static validateRequiredFields(user) {
        const fields = [
            'name',
            'email',
            'password'
        ];
        for (const field of fields) {
            if (!user[field]) {
                throw new AppError(`Missing required field: ${field}`, 400);
            }
        }
    }
}
exports.User = User;
