"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserService = void 0;
const user_entity_1 = require("./user.entity");
const bcrypt_service_1 = require("../../shared/services/bcrypt.service");
/**
 * Service responsible for business logic related to Users.
 * Handles creation, retrieval, update, and deletion of users.
 */
class UserService {
    constructor({ userRepository }) {
        /**
         * Create a new user
         * @param params - The user data to create
         * @returns The created user document
         */
        this.createUser = (params) => __awaiter(this, void 0, void 0, function* () {
            const newUser = new user_entity_1.User({
                name: params.name,
                email: params.email,
                password: params.password,
                roles: []
            });
            const existingUser = yield this.userRepository.findUserByEmail(params.email);
            if (existingUser)
                throw new AppError('Email already in use', 400);
            const userData = newUser.getData();
            // Hash the password
            userData.password = this.bcryptService.hashPassword(params.password);
            return yield this.userRepository.createUser(userData);
        });
        /**
         * Assign a role to a user
         * @param userId - The user's ID
         * @param roleIds - The role's IDs
         * @returns The user document
         */
        this.assignRoleToUser = (userId, roleIds) => __awaiter(this, void 0, void 0, function* () {
            if (!userId || !roleIds) {
                throw new AppError('User Id or role Id not provided', 400);
            }
            const user = yield this.userRepository.findUserById(userId);
            if (!user) {
                throw new AppError('User not found', 404);
            }
            const existingRolesIds = user.roles || [];
            const rolesToAdd = roleIds.filter(roleId => !existingRolesIds.includes(roleId));
            if (rolesToAdd.length === 0) {
                throw new AppError('User already have these role', 400);
            }
            const existingRoles = yield this.listRolesDep({ _id: { $in: roleIds } });
            const foundRoleIds = existingRoles.map(role => role.id);
            const newRoles = [
                ...existingRolesIds,
                ...foundRoleIds
            ];
            if (user.roles === newRoles) {
                throw new AppError('There are no roles to be added', 400);
            }
            const newUser = new user_entity_1.User(Object.assign(Object.assign({}, user), { roles: newRoles }));
            return yield this.userRepository.updateUserById(userId, newUser.getData());
        });
        /**
         * Get a user by ID
         * @param id - The user's ID
         * @returns The user document or null if not found
         */
        this.getUserById = (id) => __awaiter(this, void 0, void 0, function* () {
            const user = yield this.userRepository.findUserById(id);
            if (!user)
                throw new AppError('User not found', 404);
            return user;
        });
        /**
         * Get a user by email
         * @param email - The user's email
         * @returns The user document or null if not found
         */
        this.getUserByEmail = (email) => __awaiter(this, void 0, void 0, function* () {
            const user = yield this.userRepository.findUserByEmail(email);
            if (!user)
                throw new AppError('User not found', 404);
            return user;
        });
        /**
         * Get all roles of user
         * @param id - User's id
         * @returns Array of roles
         */
        this.getRolesFromUserById = (id) => __awaiter(this, void 0, void 0, function* () {
            const user = yield this.userRepository.findUserById(id);
            if (!user)
                throw new AppError('User not found', 404);
            if (!user.roles || user.roles.length === 0) {
                throw new AppError('User does not have roles', 400);
            }
            // Awaits the resolution of all promisses
            return yield this.listRolesDep({ _id: { $in: user.roles } });
        });
        /**
         * List all users with optional filters
         * @param filter - Filters for the query
         * @returns An array of user documents
         */
        this.listUsers = (...args_1) => __awaiter(this, [...args_1], void 0, function* (filter = {}) {
            return yield this.userRepository.listUsers(filter);
        });
        /**
         * Update a user's information by ID
         * @param params.id - The user's ID
         * @param params.userData - The data to update
         * @returns The updated user document or null if not found
         */
        this.updateUserById = (params) => __awaiter(this, void 0, void 0, function* () {
            const user = yield this.userRepository.findUserById(params.id);
            if (!user)
                throw new AppError('User not found', 404);
            // Merge existing user data with new data to validate
            const newUser = new user_entity_1.User(Object.assign(Object.assign({}, user), params.userData));
            const userData = newUser.getData();
            // Hash the new password before updating
            if (params.userData.password) {
                userData.password = this.bcryptService.hashPassword(params.userData.password);
            }
            return yield this.userRepository.updateUserById(params.id, userData);
        });
        /**
         * Remove one or more roles of user
         * @param userId - User's ID
         * @param roleId - Role's ID
         * @returns
         */
        this.removeRoleFromUserById = (userId, roleIds) => __awaiter(this, void 0, void 0, function* () {
            var _a;
            const user = yield this.userRepository.findUserById(userId);
            if (!user)
                throw new AppError('User not found', 404);
            if (!user.roles || user.roles.length === 0) {
                throw new AppError('User does not have roles', 400);
            }
            const rolesToRemove = roleIds.filter(roleId => { var _a; return (_a = user.roles) === null || _a === void 0 ? void 0 : _a.includes(roleId); });
            if (rolesToRemove.length === 0) {
                throw new AppError('User no longer has these roles', 400);
            }
            const roles = yield this.listRolesDep({ _id: { $in: roleIds } });
            if (roles.length === 0) {
                throw new AppError("Roles not found", 404);
            }
            const userRoles = (_a = user.roles) === null || _a === void 0 ? void 0 : _a.filter(role => !rolesToRemove.includes(role));
            const newUser = new user_entity_1.User(Object.assign(Object.assign({}, user), { roles: userRoles }));
            return yield this.userRepository.updateUserById(userId, newUser.getData());
        });
        /**
         * Delete a user by ID
         * @param id - The user's ID
         * @returns The deleted user document or null if not found
         */
        this.deleteUserById = (id) => __awaiter(this, void 0, void 0, function* () {
            const user = yield this.userRepository.findUserById(id);
            if (!user)
                throw new AppError('User not found', 404);
            return yield this.userRepository.deleteUserById(id);
        });
        this.createStudent = (params) => __awaiter(this, void 0, void 0, function* () {
            return yield this.createStudentDep(params);
        });
        this.getStudentById = (id) => __awaiter(this, void 0, void 0, function* () {
            return yield this.getStudentByIdDep(id);
        });
        this.listStudents = (filter) => __awaiter(this, void 0, void 0, function* () {
            return yield this.listStudentsDep(filter);
        });
        this.updateStudent = (params) => __awaiter(this, void 0, void 0, function* () {
            return yield this.updateStudentDep(params);
        });
        this.deleteStudent = (id) => __awaiter(this, void 0, void 0, function* () {
            return yield this.deleteStudentDep(id);
        });
        this.createTeacher = (params) => __awaiter(this, void 0, void 0, function* () {
            return yield this.createTeacherDep(params);
        });
        this.getTeacherById = (id) => __awaiter(this, void 0, void 0, function* () {
            return yield this.getTeachersDep(id);
        });
        this.listTeachers = (filter) => __awaiter(this, void 0, void 0, function* () {
            return yield this.listTeachersDep(filter);
        });
        this.updateTeacher = (params) => __awaiter(this, void 0, void 0, function* () {
            return yield this.updateTeacherDep(params);
        });
        this.deleteTeacher = (id) => __awaiter(this, void 0, void 0, function* () {
            return yield this.deleteTeacherDep(id);
        });
        this.userRepository = userRepository;
        this.bcryptService = new bcrypt_service_1.BcryptService();
    }
}
exports.UserService = UserService;
