"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BcryptService = void 0;
const bcryptjs_1 = __importDefault(require("bcryptjs"));
class BcryptService {
    constructor() {
        this.saltsRounds = Number(process.env.CRYPT_SALT_ROUNDS);
        if (!this.saltsRounds) {
            throw new AppError('Environment variables not found', 500);
        }
    }
    /**
     * Hash a password
     * @param password - The password to hash
     * @returns The hashed password
     */
    hashPassword(password) {
        if (this.isHash(password))
            return password;
        return bcryptjs_1.default.hashSync(password, this.saltsRounds);
    }
    /**
     * Verify if a password matches the secret
     * @param password - The password to verify
     * @param hash - The hashed password to compare against
     * @returns - True if password matches the secret, false otherwise
     */
    validatePassword(password, hash) {
        return bcryptjs_1.default.compareSync(password, hash);
    }
    /**
     * Checks if the given string is already a hash.
     * @param str - The string to check.
     * @returns `true` if the string is a hash, `false` otherwise.
     */
    isHash(str) {
        if (typeof str !== 'string' || str.length !== 60) {
            return false;
        }
        const regex = /^\$2[aby]\$\d{2}\$/;
        return regex.test(str);
    }
}
exports.BcryptService = BcryptService;
