"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StreamingService = void 0;
class StreamingService {
    constructor() {
        this.token = process.env.VIDEOCHAMADA_API_TOKEN;
        this.url = `https://api.videochamada.com.br/api/calls`;
        if (!this.token)
            throw new AppError('Environment variables not found', 500);
    }
    /**
     * Creates a new streaming call session.
     *
     * @param {Object} data - Call creation payload.
     * @param {string} data.title - Title of the call.
     * @param {string} data.description - Description of the call.
     * @param {Date} data.expiresAt - Expiration date for the call.
     *
     * @returns {Promise<string>} - The generated call URL.
     *
     * @throws {Error} If the API request fails.
     */
    createCall(data) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this._request({
                method: 'POST',
                endpoint: '',
                body: JSON.stringify(data),
            });
            return response;
        });
    }
    /**
     * Lists all calls within a specific date range.
     *
     * @param {Object} data - Filter parameters.
     * @param {string} data.startDate - ISO date marking the beginning of the range.
     * @param {string} data.endDate - ISO date marking the end of the range.
     *
     * @returns {Promise<ICall[]>} - Array of call objects.
     *
     * @throws {Error} If the API request fails.
     */
    listCalls(data) {
        return __awaiter(this, void 0, void 0, function* () {
            const calls = yield this._request({
                method: 'GET',
                endpoint: `?${new URLSearchParams(data).toString()}`
            });
            return calls.data;
        });
    }
    /**
     * Marks a call as ended and requests deletion.
     *
     * @param {string} id - The ID of the call to be ended.
     *
     * @returns {Promise<void>}
     *
     * @throws {Error} If the API request fails.
     */
    deleteCall(id) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this._request({
                method: 'POST',
                endpoint: `/${id}/end`
            });
            if (response.statusCode === 400)
                return;
        });
    }
    /**
     * Performs an HTTP request to the VideoChamada API.
     *
     * @private
     * @param {Object} params - Request configuration.
     * @param {'GET' | 'POST'} params.method - HTTP method.
     * @param {string} params.endpoint - API endpoint suffix.
     * @param {BodyInit} [params.body] - Optional request body.
     *
     * @returns {Promise<ICall | IListCalls | IDeleteCall>} - Parsed JSON response.
     *
     * @throws {Error} If the HTTP request fails.
     */
    _request(params) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const response = yield fetch(`${this.url}${params.endpoint}`, {
                    headers: {
                        'Authorization': `Bearer ${this.token}`,
                        'Content-Type': 'application/json',
                    },
                    method: params.method,
                    body: params.body,
                });
                return yield response.json();
            }
            catch (err) {
                throw new AppError(`Class could not be scheduled: ${err.message}`, 500);
            }
        });
    }
}
exports.StreamingService = StreamingService;
