#!/usr/bin/env node
import { fileURLToPath } from 'url';
import { dirname, join } from 'path';
import { existsSync, mkdirSync, readFileSync, writeFileSync } from 'fs';

// Configuration - using import.meta.url to get current directory
const currentFileUrl = import.meta.url;
const currentFilePath = fileURLToPath(currentFileUrl);
const currentDirPath = dirname(currentFilePath);

// Configuration
const CONFIG = {
    templatesDir: join(currentDirPath, 'templates'),
    fileStructure: [
        {
            template: './infrastructure/repository.ts.template',
            output: 'src/infrastructure/repository/{moduleName}.repository.ts'
        },
        {
            template: './infrastructure/model.ts.template',
            output: 'src/infrastructure/database/models/{moduleName}.model.ts'
        },
        {
            template: './service/interface.ts.template',
            output: 'src/modules/{ModuleName}/interfaces/{moduleName}.interface.ts'
        },
        {
            template: './service/repository.interface.ts.template',
            output: 'src/modules/{ModuleName}/interfaces/{moduleName}.repository.interface.ts'
        },
        {
            template: './service/entity.ts.template',
            output: 'src/modules/{ModuleName}/{moduleName}.entity.ts'
        },
        {
            template: './service/service.ts.template',
            output: 'src/modules/{ModuleName}/{moduleName}.service.ts'
        },
        {
            template: './application/service.interface.ts.template',
            output: 'src/modules/{ModuleName}/interfaces/{moduleName}.service.interface.ts'
        },
        {
            template: './application/controller.ts.template',
            output: 'src/modules/{ModuleName}/{moduleName}.controller.ts'
        },
        {
            template: './application/router.ts.template',
            output: 'src/main/routes/{moduleName}.router.ts'
        },
        {
            template: './factories/service.factory.ts.template',
            output: 'src/main/factories/{moduleName}/{moduleName}.service.factory.ts'
        },
        {
            template: './factories/controller.factory.ts.template',
            output: 'src/main/factories/{moduleName}/{moduleName}.controller.factory.ts'
        },
        {
            template: './factories/router.factory.ts.template',
            output: 'src/main/factories/{moduleName}/{moduleName}.router.factory.ts'
        },
        {
            template: './docs/docs.yaml.template',
            output: 'src/docs/app/{moduleName}.docs.yaml'
        },
        {
            template: './docs/create.component.yaml.template',
            output: 'src/docs/components/create/{moduleName}.create.component.yaml'
        },
        {
            template: './docs/update.component.yaml.template',
            output: 'src/docs/components/update/{moduleName}.update.component.yaml'
        },
        {
            template: './docs/response.component.yaml.template',
            output: 'src/docs/components/responses/{moduleName}.response.component.yaml'
        },
    ]
};

// Case conversion utilities
const CaseConverter = {
    // Normalize any format to base format (snake_case)
    normalizeInput: (str) => {
        return str
            .trim()                             // Remove extra spaces
            .replace(/([A-Z])/g, '_$1')         // PascalCase/camelCase to snake_case
            .replace(/[- ]/g, '_')              // kebab-case and space case to snake_case
            .toLowerCase()                      // Convert everything to lowercase
            .replace(/^_+|_+$/g, '')            // Remove leading and trailing underscores
            .replace(/_+/g, '_');               // Remove duplicate underscores
    },

    toPascalCase: (str) => {
        const normalized = CaseConverter.normalizeInput(str);
        return normalized
            .split('_')
            .map(word => word.charAt(0).toUpperCase() + word.slice(1))
            .join('');
    },

    toCamelCase: (str) => {
        const normalized = CaseConverter.normalizeInput(str);
        return normalized
            .split('_')
            .map((word, index) => 
                index === 0 
                    ? word.toLowerCase() 
                    : word.charAt(0).toUpperCase() + word.slice(1)
            )
            .join('');
    },

    toSnakeCase: (str) => {
        return CaseConverter.normalizeInput(str);
    },

    toKebabCase: (str) => {
        const normalized = CaseConverter.normalizeInput(str);
        return normalized.replace(/_/g, '-');
    },

    toSpacedName: (str) => {
        const normalized = CaseConverter.normalizeInput(str);
        return normalized.replace(/_/g, ' ');
    },

    toSpacedUpperName: (str) => {
        const normalized = CaseConverter.normalizeInput(str);
        return normalized
            .split('_')
            .map(word => word.charAt(0).toUpperCase() + word.slice(1))
            .join(' ');
    },
};

// Function to process variables in templates
function processTemplate(content, moduleName) {
    const moduleNames = {
        '{moduleName}': CaseConverter.toCamelCase(moduleName),
        '{module-name}': CaseConverter.toKebabCase(moduleName),
        '{ModuleName}': CaseConverter.toPascalCase(moduleName),
        '{module_name}': CaseConverter.toSnakeCase(moduleName),
        '{module name}': CaseConverter.toSpacedName(moduleName),
        '{Module Name}': CaseConverter.toSpacedUpperName(moduleName),
        '{MODULE_NAME}': CaseConverter.toSnakeCase(moduleName).toUpperCase(),
    };

    let processedContent = content;
    Object.entries(moduleNames).forEach(([placeholder, value]) => {
        const regex = new RegExp(placeholder.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'g');
        processedContent = processedContent.replace(regex, value);
    });

    return processedContent;
}

// Function to create directories recursively
function ensureDirectoryExistence(filePath) {
    const directory = dirname(filePath);
    if (existsSync(directory)) {
        return true;
    }
    ensureDirectoryExistence(directory);
    mkdirSync(directory, { recursive: true });
}

// Function to load template
function loadTemplate(templateName) {
    const templatePath = join(CONFIG.templatesDir, templateName);
    
    if (!existsSync(templatePath)) {
        throw new Error(`Template not found: ${templatePath}`);
    }
    
    return readFileSync(templatePath, 'utf8');
}

// Main function
function createModule(moduleName) {
    let createdFiles = 0;
    let skippedFiles = 0;

    CONFIG.fileStructure.forEach(({ template, output }) => {
        try {
            const outputPath = processTemplate(output, moduleName);
            
            if (existsSync(outputPath)) {
                console.log(`⚠️ File already exists: ${outputPath}`);
                skippedFiles++;
                return;
            }

            const templateContent = loadTemplate(template);
            const processedContent = processTemplate(templateContent, moduleName);
            
            ensureDirectoryExistence(outputPath);
            writeFileSync(outputPath, processedContent);
            
            console.log(`✅ Created: ${outputPath}`);
            createdFiles++;
        
        } catch (error) {
            console.error(`❌ Error processing ${template}:`, error.message);
        }
    });

    return { createdFiles, skippedFiles };
}

// Input validation
function validateInput() {
    if (process.argv.length < 3) {
        console.error('❌ Please provide module name: npm run create:module {moduleName}');
        console.error('   Example: npm run create:module user');
        console.error('   Example: npm run create:module product_category');
        console.error('   Example: npm run create:module ProductCategory');
        console.error('   Example: npm run create:module product-category');
        console.error('   Example: npm run create:module "product category"');
        process.exit(1);
    }

    const moduleName = process.argv[2];
    
    // More flexible validation - accepts letters, numbers, spaces, hyphens and underscores
    if (!/^[a-zA-Z0-9\s\-_]+$/.test(moduleName)) {
        console.error('❌ Invalid module name. Use only letters, numbers, spaces, hyphens and underscores.');
        process.exit(1);
    }

    return moduleName;
}

// Script execution
(function(){
    try {
        const moduleName = validateInput();
        const { createdFiles, skippedFiles } = createModule(moduleName);

        console.log(`\n🎉 Module created successfully!`);
        console.log(`📊 Statistics:`);
        console.log(`✅ ${createdFiles} files created`);
        console.log(`⚠️  ${skippedFiles} files already existed`);

        console.log(`\nNext steps:`);
        console.log(`1. Adapt interface and model with required properties.`);
        console.log(`2. Create business logic inside service.`);
        console.log(`3. Add RouterFactory inside index.ts .`);
        console.log(`4. Create documentation inside "docs" to use with swagger.\n`);

    } catch (error) {
        console.error('❌ Error during execution:', error.message);
        process.exit(1);
    }
})()