import mongoose from "mongoose";

export class DatabaseService {
    private protocol?: string;
    private host?: string;
    private port?: string;
    private username?: string;
    private password?: string;
    private database?: string;
    private databaseURI: string;

    constructor() {
        this.protocol = process.env.DB_PROTOCOL;
        this.host = process.env.DB_HOST;
        this.port = process.env.DB_PORT || '27017';
        this.username = process.env.DB_USERNAME;
        this.password = process.env.DB_PASSWORD;
        this.database = process.env.DB_DATABASE;

        // Ensure complete connection string
        if (!this.protocol || !this.host || !this.database || !this.username || !this.password) {
            throw new AppError('Incomplete connection string', 500);
        }

        if (this.protocol == 'mongodb+srv') {
            this.databaseURI = this._buildAtlasURI();
        } else {
            this.databaseURI = this._buildLocalURI();
        }
    }

    /**
     * Connects to database
     */
    public async connect(): Promise<void> {
        try {
            await mongoose.connect(this.databaseURI);
        } catch(error) {
            throw new AppError(`Failed to connect to MongoDB: ${(error as Error).message}`, 500);
        }
    }

    /**
     * Close database connection
     */
    public async disconnect() {
        await mongoose.disconnect();
    }

    /**
     * Create connection string for localhost
     */
    private _buildLocalURI():string {
        return `${this.protocol}://${this.username}:${this.password}@${this.host}:${this.port}/${this.database}`;
    }

    /**
     * Create connection string for atlas
     */
    private _buildAtlasURI():string {
        return `${this.protocol}://${this.username}:${this.password}@${this.host}/${this.database}?retryWrites=true&w=majority`;
    }
}