import { ICategory, IMCategory } from "../../modules/Category/interfaces/category.interface";
import { ICategoryRepository } from "../../modules/Category/interfaces/category.repository.interface";
import { MCategory } from "../database/models/category.model";

export class CategoryRepository implements ICategoryRepository {
    /**
     * Creates a new category in the database
     * @param category - Category data to be created
     * @returns The created category
     */
    public async createCategory(
        category: ICategory
    ): Promise<ICategory> {
        const newCategory = await MCategory.create(category);
        return this.adaptor(newCategory) as ICategory;
    }

    /**
     * Finds a category by ID
     * @param id - Category ID
     * @returns The found category or null
     */
    public async findCategoryById(
        id: string
    ): Promise<ICategory | null> {
        const category = await MCategory.findOne({ _id: id });
        return this.adaptor(category);
    }

    /**
     * Finds a category by name
     * @param name - Category name
     * @returns The found category or null
     */
    public async findCategoryByName(
        name: string
    ): Promise<ICategory | null> {
        const category = await MCategory.findOne({ name });
        return this.adaptor(category);
    }

    /**
     * Lists categorys according to filter
     * @param filter - Search filter
     * @returns Array of found categorys
     */
    public async listCategories(
        filter: Partial<ICategory>
    ): Promise<ICategory[] | []> {
        const categorys = await MCategory.find(filter);
        return categorys.map(category =>
            this.adaptor(category)
        ) as ICategory[];
    }

    /**
     * Updates a category by ID
     * @param id - Category ID
     * @param data - Data for update
     * @returns The updated category or null
     */
    public async updateCategoryById(
        id: string,
        data: Partial<ICategory>
    ): Promise<ICategory | null> {
        const category = await MCategory.findOneAndUpdate(
            { _id: id },
            { $set: data },
            { new: true },
        );
        return this.adaptor(category);
    }

    /**
     * Deletes a category by ID
     * @param id - Category ID
     * @returns The deleted category or null
     */
    public async deleteCategoryById(
        id: string
    ): Promise<ICategory | null> {
        const category = await MCategory.findByIdAndDelete(id);
        return this.adaptor(category);
    }

    /**
     * Adapts the mongoose document to the ICategory model
     * @param category - Mongoose document
     * @returns The adapted category or null
     */
    private adaptor(
        category: IMCategory | null
    ): ICategory | null {
        if (!category) return null;
        const { _id, __v, ...response } = category.toObject();
        return {
            id: _id.toString(),
            ...response
        } as ICategory;
    }
}