import { FilterQuery } from "mongoose";
import { IClass, IMClass } from "../../modules/Class/interfaces/class.interface";
import { IClassFilter } from "../../modules/Class/interfaces/class.service.interface";
import { MClass } from "../database/models/class.model";
import { IClassRepository } from "../../modules/Class/interfaces/class.repository.interface";

export class ClassRepository implements IClassRepository {
    /**
     * Creates a new class in the database
     * @param class - Class data to be created
     * @returns The created class
     */
    public async createClass(
        classData: IClass
    ): Promise<IClass> {
        const newClass = await MClass.create(classData);
        return this.adaptor(newClass) as IClass;
    }

    /**
     * Finds a class by ID
     * @param id - Class ID
     * @returns The found class or null
     */
    public async findClassById(
        id: string
    ): Promise<IClass | null> {
        const classData = await MClass.findOne({ _id: id });
        return this.adaptor(classData);
    }

    /**
     * Lists classs according to filter
     * @param filter - Search filter
     * @returns Array of found classs
     */
    public async listClasses(
        filter: Partial<IClass>
    ): Promise<IClass[] | []> {
        const query = this.configureQueryObject(filter);
        const classes = await MClass.find(query);
        return classes.map(classData =>
            this.adaptor(classData)
        ) as IClass[];
    }

    /**
     * Updates a class by ID
     * @param id - Class ID
     * @param data - Data for update
     * @returns The updated class or null
     */
    public async updateClassById(
        id: string,
        data: Partial<IClass>
    ): Promise<IClass | null> {
        const classData = await MClass.findOneAndUpdate(
            { _id: id },
            { $set: data },
            { new: true },
        );
        return this.adaptor(classData);
    }

    /**
     * Deletes a class by ID
     * @param id - Class ID
     * @returns The deleted class or null
     */
    public async deleteClassById(
        id: string
    ): Promise<IClass | null> {
        const classData = await MClass.findByIdAndDelete(id);
        return this.adaptor(classData);
    }

    /**
     * Adapts the mongoose document to the IClass model
     * @param class - Mongoose document
     * @returns The adapted class or null
     */
    private adaptor(
        classData: IMClass | null
    ): IClass | null {
        if (!classData) return null;
        const { _id, __v, ...response } = classData.toObject();
        return {
            id: _id.toString(),
            ...response
        } as IClass;
    }

    private configureQueryObject(
        query: IClassFilter
    ): FilterQuery<IClass> {
        const filter: FilterQuery<IClass> = {};

        if (query.topic_id) {
            filter.topic_id = query.topic_id;
        }
        if (query.teacher_id) {
            filter.teacher_id = query.teacher_id;
        }
        if (query.status) {
            filter.status = query.status;
        }
        if (query.type) {
            filter.type = query.type;
        }
        if (query.payment_status) {
            filter.payment_status = query.payment_status;
        }
        if (query.student_id) {
            filter.main_student_id = query.student_id;
        }
        if (query.start_time) {
            filter.start_time = { $lt: query.end_time };
        }
        if (query.end_time) {
            filter.end_time = { $gt: query.start_time };
        }

        return filter;
    }
}