import { IDiscipline, IMDiscipline } from "../../modules/Discipline/interfaces/discipline.interface";
import { MDiscipline } from "../database/models/discipline.model";

export class DisciplineRepository {
    /**
     * Creates a new discipline in the database
     * @param discipline - Discipline data to be created
     * @returns The created discipline
     */
    public async createDiscipline(
        discipline: IDiscipline
    ): Promise<IDiscipline> {
        const newDiscipline = await MDiscipline.create(discipline);
        return this.adaptor(newDiscipline) as IDiscipline;
    }

    /**
     * Finds a discipline by ID
     * @param id - Discipline ID
     * @returns The found discipline or null
     */
    public async findDisciplineById(
        id: string
    ): Promise<IDiscipline | null> {
        const discipline = await MDiscipline.findOne({ _id: id });
        return this.adaptor(discipline);
    }

    /**
     * Finds a discipline by name
     * @param name - Discipline name
     * @returns The found discipline or null
     */
    public async findDisciplineByName(
        name: string
    ): Promise<IDiscipline | null> {
        const discipline = await MDiscipline.findOne({ name });
        return this.adaptor(discipline);
    }

    /**
     * Lists disciplines according to filter
     * @param filter - Search filter
     * @returns Array of found disciplines
     */
    public async listDisciplines(
        filter: Partial<IDiscipline>
    ): Promise<IDiscipline[] | []> {
        const disciplines = await MDiscipline.find(filter);
        return disciplines.map(discipline =>
            this.adaptor(discipline)
        ) as IDiscipline[];
    }

    /**
     * Updates a discipline by ID
     * @param id - Discipline ID
     * @param data - Data for update
     * @returns The updated discipline or null
     */
    public async updateDisciplineById(
        id: string,
        data: Partial<IDiscipline>
    ): Promise<IDiscipline | null> {
        const discipline = await MDiscipline.findOneAndUpdate(
            { _id: id },
            { $set: data },
            { new: true },
        );
        return this.adaptor(discipline);
    }

    /**
     * Deletes a discipline by ID
     * @param id - Discipline ID
     * @returns The deleted discipline or null
     */
    public async deleteDisciplineById(
        id: string
    ): Promise<IDiscipline | null> {
        const discipline = await MDiscipline.findByIdAndDelete(id);
        return this.adaptor(discipline);
    }

    /**
     * Adapts the mongoose document to the IDiscipline model
     * @param discipline - Mongoose document
     * @returns The adapted discipline or null
     */
    private adaptor(
        discipline: IMDiscipline | null
    ): IDiscipline | null {
        if (!discipline) return null;
        const { _id, __v, ...response } = discipline.toObject();
        return {
            id: _id.toString(),
            ...response
        } as IMDiscipline;
    }
}