import { IRole, IMRole } from "../../modules/Roles/interfaces/roles.interface";
import { IRoleRepository } from "../../modules/Roles/interfaces/roles.repository.interface";
import { MRole } from "../database/models/roles.model";

export class RolesRepository implements IRoleRepository {
    /**
     * Create a new role in the database
     * @param rolesData - The role data to create
     * @returns The created role document
     */
    public async createRole(
        rolesData: IRole
    ): Promise<IRole> {
        const newRole = await MRole.create(rolesData);
        return this.adaptor(newRole) as IRole;
    }

    /**
     * Find a role by ID
     * @param id - The role's ID
     * @returns The role document or null if not found
     */
    public async findRoleById(
        id: string
    ): Promise<IRole | null> {
        const role = await MRole.findOne({ _id: id });
        return this.adaptor(role);
    }

    /**
     * Find a role by name
     * @param name - The role's name
     * @returns The role document or null if not found
     */
    public async findRoleByName(
        name: string
    ): Promise<IRole | null> {
        const role = await MRole.findOne({ name });
        return this.adaptor(role);
    }

    /**
     * List all roles
     * @param filter - Filter to apply
     * @returns - An array of roles documents
     */
    public async listRole(
        filter: object
    ): Promise<IRole[] | []> {
        const roles = await MRole.find(filter);
        return roles.map(role =>
            this.adaptor(role)
        ) as IRole[] | [];
    }

    /**
     * Update a role by ID
     * @param id - The role's ID
     * @param updateData - The data to update
     * @returns The updated role document or null if not found
     */
    public async updateRoleById(
        id: string,
        updateData: Partial<IRole>,
    ): Promise<IRole | null> {
        const role = await MRole.findOneAndUpdate(
            { _id: id }, // filter
            { $set: updateData }, // partial update
            { new: true }, // options: return the updated document
        );
        return this.adaptor(role);
    }

    /**
     * Delete a role by ID
     * @param id - The role's ID
     * @returns The deleted role document or null if not found
     */
    public async deleteRoleById(
        id: string
    ): Promise<IRole | null> {
        const role = await MRole.findOneAndDelete({ _id: id });
        return this.adaptor(role);
    }

    /**
     * Adapts a MRole(mongoose) document to a IRole document (JSON)
     * @param role - The Mrole document
     * @returns The adapted Irole document or null if not found
     */
    private adaptor(
        role: IMRole | null
    ): IRole | null {
        if (!role) return null;
        const { _id, __v, ...response } = role.toObject();
        return {
            id: _id.toString(),
            ...response,
        } as IRole;
    }
}