import { IStudent, IMStudent } from "../../modules/Student/interfaces/student.interface";
import { MStudent } from "../database/models/student.model";

export class StudentRepository {
    /**
     * Creates a new student in the database
     * @param student - Student data to be created
     * @returns The created student
     */
    public async createStudent(
        student: IStudent
    ): Promise<IStudent> {
        const newStudent = await MStudent.create(student);
        return this.adaptor(newStudent) as IStudent;
    }

    /**
     * Finds a student by ID
     * @param id - Student ID
     * @returns The found student or null
     */
    public async getStudentById(
        id: string
    ): Promise<IStudent | null> {
        const student = await MStudent.findOne({ _id: id });
        return this.adaptor(student);
    }

    /**
     * Finds a student by email
     * @param email - Student email
     * @returns The found student or null
     */
    public async getStudentByEmail(
        email: string
    ): Promise<IStudent | null> {
        const student = await MStudent.findOne({ email });
        return this.adaptor(student);
    }

    /**
     * Lists students according to filter
     * @param filter - Search filter
     * @returns Array of found students
     */
    public async listStudents(
        filter: Partial<IStudent>
    ): Promise<IStudent[] | []> {
        const students = await MStudent.find(filter);
        return students.map(student =>
            this.adaptor(student)
        ) as IStudent[];
    }

    /**
     * Updates a student by ID
     * @param id - Student ID
     * @param data - Data for update
     * @returns The updated student or null
     */
    public async updateStudent(
        id: string,
        data: Partial<IStudent>
    ): Promise<IStudent | null> {
        const student = await MStudent.findOneAndUpdate(
            { _id: id },
            { $set: data },
            { new: true },
        );
        return this.adaptor(student);
    }

    /**
     * Deletes a student by ID
     * @param id - Student ID
     * @returns The deleted student or null
     */
    public async deleteStudent(
        id: string
    ): Promise<IStudent | null> {
        const student = await MStudent.findByIdAndDelete(id);
        return this.adaptor(student);
    }

    /**
     * Adapts the mongoose document to the IStudent model
     * @param student - Mongoose document
     * @returns The adapted student or null
     */
    private adaptor(
        student: IMStudent | null
    ): IStudent | null {
        if (!student) return null;
        const { _id, __v, ...response } = student.toObject();
        return {
            id: _id.toString(),
            ...response
        } as IStudent;
    }
}