import { PipelineStage } from "mongoose";
import { ITeacher, IMTeacher } from "../../modules/Teacher/interfaces/teacher.interface";
import { ITeacherRepository } from "../../modules/Teacher/interfaces/teacher.repository.interface";
import { MTeacher } from "../database/models/teacher.model";

export class TeacherRepository implements ITeacherRepository {
    /**
     * Get the model name in MongoDB
     * @returns - The model name
     */
    public getClassName(): string {
        return MTeacher.collection.name;
    }

    /**
     * Creates a new teacher in the database
     * @param teacher - teacher data to be created
     * @returns The created teacher
     */
    public async createTeacher(
        teacher: ITeacher
    ): Promise<ITeacher> {
        const newTeacher = await MTeacher.create(teacher);
        return this.adaptor(newTeacher) as ITeacher;
    }

    /**
     * Finds a teacher by ID
     * @param id - teacher ID
     * @returns The found teacher or null
     */
    public async getTeacherById(
        id: string
    ): Promise<ITeacher | null> {
        const teacher = await MTeacher.findOne({ _id: id });
        return this.adaptor(teacher);
    }

    /**
     * Finds a teacher by email
     * @param email - teacher email
     * @returns The found teacher or null
     */
    public async getTeacherByEmail(
        email: string
    ): Promise<ITeacher | null> {
        const teacher = await MTeacher.findOne({ email });
        return this.adaptor(teacher);
    }

    /**
     * Lists teachers according to filter
     * @param filter - Search filter
     * @returns Array of found teachers
     */
    public async listTeachers(
        filter: object
    ): Promise<ITeacher[] | []>{
        const teachers = await MTeacher.find(filter);
        return teachers.map(teacher =>
            this.adaptor(teacher)
        ) as ITeacher[] | [];
    }

    public async findTeachersWithAggregate(
        pipeline: PipelineStage[]
    ): Promise<ITeacher[]> {
        const teachers = await MTeacher.aggregate(pipeline);
        return teachers.map(teacher => 
            this.adaptor(teacher)
        ) as ITeacher[];
    }

    /**
     * Updates a teacher by ID
     * @param id - teacher ID
     * @param data - Data for update
     * @returns The updated teacher or null
     */
    public async updateTeacherById(
        id: string,
        teacher: Partial<ITeacher>,
    ): Promise<ITeacher | null> {
        const newTeacher = await MTeacher.findOneAndUpdate(
            { _id: id },
            { $set: teacher },
            { new: true },
        );
        return this.adaptor(newTeacher);
    }

    /**
     * Deletes a teacher by ID
     * @param id - teacher ID
     * @returns The deleted teacher or null
     */
    public async deleteTeacherById(
        id: string
    ): Promise<ITeacher | null> {
        const newTeacher = await MTeacher.findByIdAndDelete(id);
        return this.adaptor(newTeacher);
    }

    /**
     * Adapts the mongoose document to the Iteacher model
     * @param teacher - Mongoose document
     * @returns The adapted teacher or null
     */
    private adaptor(
        teacher: IMTeacher | null
    ): ITeacher | null {
        if(!teacher) return null

        if (typeof teacher.toObject === 'function') {
            const { _id, __v, ...response } = teacher.toObject();
            return {
                id: _id.toString(),
                ...response
            } as ITeacher;
        } else {
            const { _id, __v, ...response } = teacher;
            return {
                id: _id.toString(),
                ...response
            } as ITeacher;
        }
    }
}