import { ITopic, IMTopic } from "../../modules/Topic/interfaces/topic.interface";
import { MTopic } from "../database/models/topic.model";

export class TopicRepository {
    /**
     * Creates a new topic in the database
     * @param topic - Topic data to be created
     * @returns The created topic
     */
    public async createTopic(
        topic: ITopic
    ): Promise<ITopic> {
        const newTopic = await MTopic.create(topic);
        return this.adaptor(newTopic) as ITopic;
    }

    /**
     * Finds a topic by ID
     * @param id - Topic ID
     * @returns The found topic or null
     */
    public async findTopicById(
        id: string
    ): Promise<ITopic | null> {
        const topic = await MTopic.findOne({ _id: id });
        return this.adaptor(topic);
    }

    /**
     * Finds a topic by name
     * @param name - Topic name
     * @returns The found topic or null
     */
    public async findTopicByName(
        name: string
    ): Promise<ITopic | null> {
        const topic = await MTopic.findOne({ name });
        return this.adaptor(topic);
    }

    /**
     * Lists topics according to filter
     * @param filter - Search filter
     * @returns Array of found topics
     */
    public async listTopics(
        filter: Partial<ITopic>
    ): Promise<ITopic[] | []> {
        const topics = await MTopic.find(filter);
        return topics.map(topic =>
            this.adaptor(topic)
        ) as ITopic[];
    }

    /**
     * Updates a topic by ID
     * @param id - Topic ID
     * @param data - Data for update
     * @returns The updated topic or null
     */
    public async updateTopicById(
        id: string,
        data: Partial<ITopic>
    ): Promise<ITopic | null> {
        const topic = await MTopic.findOneAndUpdate(
            { _id: id },
            { $set: data },
            { new: true },
        );
        return this.adaptor(topic);
    }

    /**
     * Deletes a topic by ID
     * @param id - Topic ID
     * @returns The deleted topic or null
     */
    public async deleteTopicById(
        id: string
    ): Promise<ITopic | null> {
        const topic = await MTopic.findByIdAndDelete(id);
        return this.adaptor(topic);
    }

    /**
     * Adapts the mongoose document to the ITopic model
     * @param topic - Mongoose document
     * @returns The adapted topic or null
     */
    private adaptor(
        topic: IMTopic | null
    ): ITopic | null {
        if (!topic) return null;
        const { _id, __v, ...response } = topic.toObject();
        return {
            id: _id.toString(),
            ...response
        } as IMTopic;
    }
}