import { IUser, IMUser } from '../../modules/user/interfaces/user.interface';
import { MUser } from '../database/models/user.model';

export class UserRepository {
    /**
     * Create a new user in the database
     * @param userData - The user data to create
     * @returns The created user document
     */
    public async createUser(
        userData: IUser
    ): Promise<IUser> {
        const newUser = await MUser.create(userData) as IMUser;
        return this.adaptor(newUser) as IUser;
    }

    /**
     * Find a user by ID
     * @param id - The user's ID
     * @returns The user document or null if not found
     */
    public async findUserById(
        id: string
    ): Promise<IUser | null> {
        const user = await MUser.findOne({ _id: id }) as IMUser | null;
        return this.adaptor(user) as IUser | null;
    }

    /**
     * Find a user by email
     * @param email - The user's email
     * @returns The user document or null if not found
     */
    public async findUserByEmail(
        email: string
    ): Promise<IUser | null> {
        const user = await MUser.findOne({ email }) as IMUser | null;
        return this.adaptor(user) as IUser | null;
    }

    /**
     * List all users
     * @param filter - Optional filters for the query
     * @returns An array of user documents
     */
    public async listUsers(
        filter: Partial<IUser>
    ): Promise<IUser[]> {
        const users = await MUser.find(filter);
        return users.map(user =>
            this.adaptor(user as IMUser)
        ) as IUser[];
    }

    /**
     * Update a user by ID
     * @param id - The user's ID
     * @param updateData - The data to update
     * @returns The updated user document or null if not found
     */
    public async updateUserById(
        id: string,
        updateData: Partial<IUser>,
    ): Promise<IUser | null> {
        const user = await MUser.findOneAndUpdate(
            { _id: id }, // filter
            { $set: updateData }, // partial update
            { new: true }, // options: return the updated document
        ) as IMUser | null;
        return this.adaptor(user) as IUser | null;
    }

    /**
     * Delete a user by ID
     * @param id - The user's ID
     * @returns The deleted user document or null if not found
     */
    public async deleteUserById(
        id: string
    ): Promise<IUser | null> {
        const user = await MUser.findOneAndDelete({ _id: id }) as IMUser | null;
        return this.adaptor(user) as IUser | null;
    }

    /**
     * Adapts a MUser(mongoose) document to a IUser document (JSON)
     * @param user - The MUser document
     * @returns The adapted IUser document or null if not found
     */
    private adaptor(
        user: IMUser | null
    ): IUser | null {
        if (!user) return null;
        const { _id, __v, ...response } = user.toObject();
        return {
            id: _id.toString(),
            ...response,
        } as IUser;
    }
}