import { IRouter } from "../../shared/contracts/IRouter";
import { ClassRepository } from "../../infrastructure/repository/class.repository";
import { StudentRepository } from "../../infrastructure/repository/student.repository";
import { TeacherRepository } from "../../infrastructure/repository/teacher.repository";
import { ClassServiceFactory } from "./class/class.service.factory";
import { TeacherServiceFactory } from "./teacher/teacher.service.factory";
import { StudentServiceFactory } from "./student/student.service.factory";
import { TopicServiceFactory } from "./topic/topic.service.factory";
import { TeacherControllerFactory } from "./teacher/teacher.controller.factory";
import { ClassControllerFactory } from "./class/class.controller.factory";
import { StudentControllerFactory } from "./student/student.controller.factory";
import { TeacherRouterFactory } from "./teacher/teacher.router.factory";
import { StudentRouterFactory } from "./student/student.router.factory";
import { ClassRouterFactory } from "./class/class.router.factory";
import { TopicRepository } from "../../infrastructure/repository/topic.repository";
import { UserRepository } from "../../infrastructure/repository/user.repository";
import { RolesRepository } from "../../infrastructure/repository/roles.repository";
import { CategoryRepository } from "../../infrastructure/repository/category.repository";
import { DisciplineRepository } from "../../infrastructure/repository/discipline.repository";
import { UserServiceFactory } from "./user/user.service.factory";
import { RolesServiceFactory } from "./roles/roles.service.factory";
import { CategoryServiceFactory } from "./category/category.service.factory";
import { DisciplineServiceFactory } from "./discipline/discipline.service.factory";
import { UserAuthServiceFactory } from "./user/auth/user.auth.service.factory";
import { StudentAuthServiceFactory } from "./student/auth/student.auth.service.factory";
import { TeacherAuthServiceFactory } from "./teacher/auth/teacher.auth.service.factory";
import { UserAuthMiddlewareFactory } from "./user/auth/user.auth.middleware.factory";
import { UserAuthControllerFactory } from "./user/auth/user.auth.controller.factory";
import { UserAuthRouterFactory } from "./user/auth/user.auth.router.factory";
import { UserControllerFactory } from "./user/user.controller.factory";
import { UserRouterFactory } from "./user/user.router.factory";
import { RolesControllerFactory } from "./roles/roles.controller.factory";
import { CategoryControllerFactory } from "./category/category.controller.factory";
import { DisciplineControllerFactory } from "./discipline/discipline.controller.factory";
import { TopicControllerFactory } from "./topic/topic.controller.factory";
import { RolesRouterFactory } from "./roles/roles.router.factory";
import { CategoryRouterFactory } from "./category/category.router.factory";
import { DisciplineRouterFactory } from "./discipline/discipline.router.factory";
import { TopicRouterFactory } from "./topic/topic.router.factory";
import { PermissionsMiddlewareFactory } from "./roles/permissions.middleware.factory";
import { StudentAuthControllerFactory } from "./student/auth/student.auth.controller.factory";
import { TeacherAuthControllerFactory } from "./teacher/auth/teacher.auth.controller.factory";
import { StudentAuthRouterFactory } from "./student/auth/student.auth.router.factory";
import { TeacherAuthRouterFactory } from "./teacher/auth/teacher.auth.router.factory";
import { TeacherAuthMiddlewareFactory } from "./teacher/auth/teacher.auth.middleware.factory";
import { StudentAuthMiddlewareFactory } from "./student/auth/student.auth.middleware.factory";

/**
 * Instantiates the modules correctly
 * @returns Array of Routers
 */
export class Container {
    public static inicialize(): IRouter[] {

        /**
         * Instantiates the repositories
         */
        const userRepo = new UserRepository();
        const rolesRepo = new RolesRepository();
        const categoryRepo = new CategoryRepository();
        const disciplineRepo = new DisciplineRepository();
        const topicRepo = new TopicRepository();
        const teacherRepo = new TeacherRepository();
        const studentRepo = new StudentRepository();
        const classRepo = new ClassRepository();

        /*
         * Instantiates the services
         */
        const userAuthService = UserAuthServiceFactory.create(userRepo);
        const userService = UserServiceFactory.create(userRepo);
        const rolesService = RolesServiceFactory.create(rolesRepo);
        const categoryService = CategoryServiceFactory.create(categoryRepo);
        const disciplineService = DisciplineServiceFactory.create(disciplineRepo);
        const topicService = TopicServiceFactory.create(topicRepo);
        const studentService = StudentServiceFactory.create(studentRepo);
        const studentAuthService = StudentAuthServiceFactory.create(studentRepo);
        const teacherService = TeacherServiceFactory.create(teacherRepo);
        const teacherAuthService = TeacherAuthServiceFactory.create(teacherRepo);
        const classService = ClassServiceFactory.create(classRepo);

        /**
         * Injecting the dependences
         */
        // User
        userService.listRolesDep = rolesService.listRoles;
        userService.createStudentDep = studentService.createStudent;
        userService.getStudentByIdDep = studentService.getStudentById;
        userService.listStudentsDep = studentService.listStudents;
        userService.updateStudentDep = studentService.updateStudent;
        userService.deleteStudentDep = studentService.deleteStudent;
        userService.createTeacherDep = teacherService.createTeacher;
        userService.getTeachersDep = teacherService.getTeacherById;
        userService.listTeachersDep = teacherService.listTeacher;
        userService.updateTeacherDep = teacherService.updateTeacherById;
        userService.deleteTeacherDep = teacherService.deleteTeacherById;

        // Discipline
        disciplineService.getCategoryById = categoryService.getCategoryById;

        // Topic
        topicService.getDisciplineById = disciplineService.getDisciplineById;

        // Teacher
        teacherService.listClassesDep = classService.listClasses;
        teacherService.getClassByIdDep = classService.getClassById;
        teacherService.updateClassByIdDep = classService.updateClassById;
        teacherService.listTopicsDep = topicService.listTopics;

        // Student
        studentService.listTeachersDep = teacherService.findTeacherForStudent;
        studentService.createClassDep = classService.createClass;
        studentService.getClassByIdDep = classService.getClassById;
        studentService.listClassesDep = classService.listClasses;
        studentService.updateClassDep = classService.updateClassById;

        // Class
        classService.listStudents = studentService.listStudents;
        classService.getStudentById = studentService.getStudentById;
        classService.getTeacherById = teacherService.getTeacherById;
        classService.getTopicById = topicService.getTopicById;

        /**
         * Inicialize the middlewares
         */
        const userAuthMiddleware = UserAuthMiddlewareFactory.create(userAuthService);
        const teacherAuthMiddleware = TeacherAuthMiddlewareFactory.create(teacherAuthService);
        const studentAuthMiddleware = StudentAuthMiddlewareFactory.create(studentAuthService);
        const permissionsMiddleware = PermissionsMiddlewareFactory.create(rolesService);

        /**
         * Instantiates the controllers
         */
        const userAuthController = UserAuthControllerFactory.create(userAuthService);
        const userController = UserControllerFactory.create(userService);
        const rolesController = RolesControllerFactory.create(rolesService);
        const categoryController = CategoryControllerFactory.create(categoryService);
        const disciplineController = DisciplineControllerFactory.create(disciplineService);
        const topicController = TopicControllerFactory.create(topicService);
        const studentAuthController = StudentAuthControllerFactory.create(studentAuthService);
        const studentController = StudentControllerFactory.create(studentService);
        const teacherAuthController = TeacherAuthControllerFactory.create(teacherAuthService);
        const teacherController = TeacherControllerFactory.create(teacherService);
        const classController = ClassControllerFactory.create(classService);

        /**
         * Instantiates the routers injecting the correct dependencies
        **/

        /**
         * Authentication routes for each entity type
         */
        const userAuthRouter = UserAuthRouterFactory.create(
            userAuthController,
            userAuthMiddleware
        );
        const studentAuthRouter = StudentAuthRouterFactory.create(
            studentAuthController,
            studentAuthMiddleware,
        );
        const teacherAuthRouter = TeacherAuthRouterFactory.create(
            teacherAuthController,
            teacherAuthMiddleware,
        );

        /**
         * CRUD functions for Roles
         * Only authenticated users have access
         */
        const rolesRouter = RolesRouterFactory.create(
            rolesController,
            userAuthMiddleware,
            permissionsMiddleware,
        );

        /**
         * CRUD functions for Categories
         */
        const categoryRouter = CategoryRouterFactory.create(
            categoryController,
            userAuthMiddleware,
            permissionsMiddleware,
        );

        /**
         * CRUD functions for Disciplines
         */
        const disciplineRouter = DisciplineRouterFactory.create(
            disciplineController,
            userAuthMiddleware,
            permissionsMiddleware,
        );

        /**
         * CRUD functions for Topics
         */
        const topicRouter = TopicRouterFactory.create(
            topicController,
            userAuthMiddleware,
            permissionsMiddleware,
        );

        /**
         * CRUD functions for Classes
         */
        const classRouter = ClassRouterFactory.create(
            classController,
        );

        /**
         * CRUD functions for Users admin
         */
        const userRouter = UserRouterFactory.create(
            userController,
            userAuthRouter,
            classRouter,
            userAuthMiddleware,
            permissionsMiddleware,
        );

        /**
         * CRUD functions for Students
         */
        const studentRouter = StudentRouterFactory.create(
            studentController,
            studentAuthRouter,
            studentAuthMiddleware,
        );

        /**
         * CRUD functions for Teachers
         */
        const teacherRouter = TeacherRouterFactory.create(
            teacherController,
            teacherAuthRouter,
            teacherAuthMiddleware,
        );

        /**
         * Export all routers
         */
        return [
            userRouter,
            rolesRouter,
            categoryRouter,
            disciplineRouter,
            topicRouter,
            studentRouter,
            teacherRouter,
        ];
    }
}