import { AuthService } from "../../../../modules/auth/auth.service";
import { AuthEntityBuilder } from "../../../../modules/auth/interfaces/auth.service.interface";
import { IStudentAuthAdapter, adaptStudentForAuth, IRegisterStudent } from "../../../../modules/Student/interfaces/student.auth.adaptor.interface";
import { IStudentRepository } from "../../../../modules/Student/interfaces/student.repository.interface";
import { IStudent } from "../../../../modules/Student/interfaces/student.interface";
import { Student } from "../../../../modules/Student/student.entity";

/**
 * Specific factory for creating AuthService for Student
 */
export class StudentAuthServiceFactory {
    /**
     * Creates an instance of AuthService configured for Student
     * @param studentRepository - Student repository
     * @returns AuthService instance for Student
     */
    public static create(
        studentRepository: IStudentRepository
    ): AuthService<
        IStudentAuthAdapter,
        IRegisterStudent
    > {
        // Adapts the Student repository to work with IStudentAuthAdapter
        const repository = {
            findById: async (
                id: string
            ): Promise<IStudentAuthAdapter | null> => {
                const student = await studentRepository.getStudentById(id);
                return student ? adaptStudentForAuth(student) : null;
            },

            findByEmail: async (
                email: string
            ): Promise<IStudentAuthAdapter | null> => {
                const student = await studentRepository.getStudentByEmail(email);
                return student ? adaptStudentForAuth(student) : null;
            },

            create: async (
                params: Partial<IStudentAuthAdapter>
            ): Promise<IStudentAuthAdapter> => {
                const student = await studentRepository.createStudent(params as unknown as IStudent);
                return adaptStudentForAuth(student);
            },

            update: async (
                id: string,
                params: Partial<IStudentAuthAdapter>
            ): Promise<IStudentAuthAdapter | null> => {
                const student = await studentRepository.updateStudent(id, params as unknown as Partial<IStudent>);
                return student ? adaptStudentForAuth(student) : null;
            }
        };

        /**
         * Constructor for new student instance
         * @param params - Required data for student registration
         * @returns Student instance
         */
        const entity: AuthEntityBuilder<
            IStudentAuthAdapter, IRegisterStudent
        > = (
            params: IRegisterStudent
        ): IStudentAuthAdapter => {
            const data: IStudent = {
                full_name: params.full_name,
                email: params.email,
                password: params.password,
                birth_date: params.birth_date,
                cpf: params.cpf,
                telephone: params.telephone,
                academic: params.academic,
                class_preferences: params.class_preferences,
                guardian: params.guardian,
                ranking: params.ranking,
                total_classes: params.total_classes,
            };

            const validatedUser = new Student(data);
            return adaptStudentForAuth(validatedUser.getData());
        }

        return new AuthService<
            IStudentAuthAdapter,
            IRegisterStudent>
        ({
            repository,
            entity
        });
    }
}