import { AuthService } from "../../../../modules/auth/auth.service";
import { AuthEntityBuilder } from "../../../../modules/auth/interfaces/auth.service.interface";
import { ITeacherAuthAdapter, adaptTeacherForAuth, IRegisterTeacher } from "../../../../modules/Teacher/interfaces/teacher.auth.adaptor.interface";
import { ITeacherRepository } from "../../../../modules/Teacher/interfaces/teacher.repository.interface";
import { ITeacher } from "../../../../modules/Teacher/interfaces/teacher.interface";
import { Teacher } from "../../../../modules/Teacher/teacher.entity";

/**
 * Factory específica para criar AuthService para Teacher
 */
export class TeacherAuthServiceFactory {
    /**
     * Cria uma instância do AuthService configurada para Teacher
     * @param teacherRepository - Repositório de professores
     * @returns Instância do AuthService para Teacher
     */
    public static create(
        teacherRepository: ITeacherRepository
    ): AuthService<
        ITeacherAuthAdapter,
        IRegisterTeacher
    > {
        // Adapta o repositório de Teacher para trabalhar com ITeacherAuthAdapter
        const repository = {
            findById: async (
                id: string
            ): Promise<ITeacherAuthAdapter | null> => {
                const teacher = await teacherRepository.getTeacherById(id);
                return teacher ? adaptTeacherForAuth(teacher) : null;
            },

            findByEmail: async (
                email: string
            ): Promise<ITeacherAuthAdapter | null> => {
                const teacher = await teacherRepository.getTeacherByEmail(email);
                return teacher ? adaptTeacherForAuth(teacher) : null;
            },

            create: async (
                params: Partial<ITeacherAuthAdapter>
            ): Promise<ITeacherAuthAdapter> => {
                const teacher = await teacherRepository.createTeacher(params as unknown as ITeacher);
                return adaptTeacherForAuth(teacher);
            },

            update: async (
                id: string,
                params: Partial<ITeacherAuthAdapter>
            ): Promise<ITeacherAuthAdapter | null> => {
                const teacher = await teacherRepository.updateTeacherById(id, params as unknown as Partial<ITeacher>);
                return teacher ? adaptTeacherForAuth(teacher) : null;
            }
        };

        /**
         * Constructor for new teacher instance
         * @param params - Required data for teacher registration
         * @returns Teacher instance
         */
        const entity: AuthEntityBuilder<
            ITeacherAuthAdapter, IRegisterTeacher
        > = (
            params: IRegisterTeacher
        ): ITeacherAuthAdapter => {
            const data: ITeacher = {
                full_name: params.full_name,
                email: params.email,
                password: params.password,
                birth_date: params.birth_date,
                cpfCnpj: params.cpfCnpj,
                rg: params.rg,
                telephone: params.telephone,
                address: params.address,
                education: params.education,
                availability: params.availability,
                preferences: params.preferences,
            };

            const validatedUser = new Teacher(data);
            return adaptTeacherForAuth(validatedUser.getData());
        }

        return new AuthService<
            ITeacherAuthAdapter,
            IRegisterTeacher
        >({ repository, entity });
    }
}
