import { AuthService } from "../../../../modules/auth/auth.service";
import { AuthEntityBuilder, IRegisterParams } from "../../../../modules/auth/interfaces/auth.service.interface";
import { IUserAuthAdapter, adaptUserForAuth } from "../../../../modules/user/interfaces/user.auth.adaptor.interface";
import { IUser } from "../../../../modules/user/interfaces/user.interface";
import { IUserRepository } from "../../../../modules/user/interfaces/user.repository.interface";
import { User } from "../../../../modules/user/user.entity";

/**
 * Specific factory for creating AuthService for User
 */
export class UserAuthServiceFactory {
    /**
     * Creates an instance of AuthService configured for User
     * @param userRepository - User repository
     * @returns AuthService instance for User
     */
    public static create(
        userRepository: IUserRepository
    ): AuthService<IUserAuthAdapter, IRegisterParams> {
        // Adapts the User repository to work with IUserAuthAdapter
        const repository = {
            
            findById: async (
                id: string
            ): Promise<IUserAuthAdapter | null> => {
                const user = await userRepository.findUserById(id);
                return user ? adaptUserForAuth(user) : null;
            },

            findByEmail: async (
                email: string
            ): Promise<IUserAuthAdapter | null> => {
                const user = await userRepository.findUserByEmail(email);
                return user ? adaptUserForAuth(user) : null;
            },
            
            create: async (
                params: Partial<IUserAuthAdapter>
            ): Promise<IUserAuthAdapter> => {
                const user = await userRepository.createUser(params as unknown as IUser);
                return adaptUserForAuth(user);
            },
            
            update: async (
                id: string,
                params: Partial<IUserAuthAdapter>
            ): Promise<IUserAuthAdapter | null> => {
                const user = await userRepository.updateUserById(id, params as unknown as Partial<IUser>);
                return user ? adaptUserForAuth(user) : null;
            }
        };

        const entity: AuthEntityBuilder<IUserAuthAdapter, IRegisterParams> = (
            params: IRegisterParams
        ): IUserAuthAdapter => {
            const userData: IUser = {
                name: params.name || "",
                email: params.email,
                password: params.password,
                roles: [] // Set a default value for creation
            };

            const validatedUser = new User(userData);
            return adaptUserForAuth(validatedUser.getData());
        }

        return new AuthService<IUserAuthAdapter, IRegisterParams>({ repository, entity });
    }
}