import { RequestHandler, Router } from 'express';
import { IRouter } from '../../shared/contracts/IRouter';
import { AuthController } from '../../modules/auth/auth.controller';
import { IAuthenticatableEntity } from '../../modules/auth/interfaces/auth.interface';
import { IRegisterParams } from '../../modules/auth/interfaces/auth.service.interface';

/**
 * Generic authentication router that can work with any authenticatable entity
 * @template T - Type of entity that extends IAuthenticatableEntity
 * @template R - Type of registration parameters that extends IRegisterParams
 */
export class AuthRouter<
    T extends IAuthenticatableEntity,
    R extends IRegisterParams
> implements IRouter {
    path = '/auth';
    router: Router;
    controller: AuthController<T, R>;
    authMiddleware: RequestHandler;

    constructor (
        controller: AuthController<T, R>,
        authMiddleware: RequestHandler,
    ) {
        this.router = Router();
        this.controller = controller;
        this.authMiddleware = authMiddleware;
        this.initRoutes();
    }

    /**
     * Initializes all authentication routes
     */
    initRoutes() {

        /**
         * Perform a new registration
         * @method POST
         * @param path - /auth/register
         */
        this.router.post(
            '/register',
            this.controller.register
        );

        /**
         * Perform a log in
         * @method POST
         * @param path - /auth/login
         */
        this.router.post(
            '/login',
            this.controller.login
        );

        /**
         * Requests password reset
         * @method POST
         * @param path - /auth/forgot-password
         */
        this.router.post(
            '/forgot-password',
            this.controller.forgotPassword
        );

        /**
         * Resets password using code
         * @method POST
         * @param path - /auth/reset-password
         */
        this.router.post(
            '/reset-password',
            this.controller.resetPassword
        );

        /**
         * View profile
         * @method GET
         * @param path - /auth/profile
         */
        this.router.get(
            '/profile',
            this.authMiddleware,
            this.controller.viewProfile
        );

        /**
         * Edit profile
         * @method PUT
         * @param path - /auth/profile
         */
        this.router.put(
            '/profile',
            this.authMiddleware,
            this.controller.editProfile
        );
    }

    /**
     * Returns the router with all routes
     */
    public getRoutes(): Router {
        return this.router;
    }
}