import { RequestHandler, Router } from "express";
import { IRouter } from "../../shared/contracts/IRouter";
import { CategoryController } from "../../modules/Category/category.controller";

/**
 * Router class for category endpoints.
 * Maps HTTP routes to CategoryController methods.
 */
export class CategoryRouter implements IRouter {
    path = '/admin/categories';
    router: Router;
    private controller: CategoryController;
    private authMiddleware: RequestHandler;
    private permissionsMiddleware: RequestHandler;

    constructor (
        controller: CategoryController,
        authMiddleware: RequestHandler,
        permissionsMiddleware: RequestHandler,
    ) {
        this.controller = controller;
        this.authMiddleware = authMiddleware;
        this.permissionsMiddleware = permissionsMiddleware;
        this.router = Router();
        this.initRoutes();
    }

    /**
     * Initializes all routes for the category controller
     * Maps endpoints to controller methods
     */
    initRoutes() {

        /**
         * Unauthenticated users can view categories
        **/

        /**
         * Lists all categorys
         * @method GET
         * @param path - /categories
         */
        this.router.get(
            '/',
            this.controller.listCategories
        );

        /**
         * Finds a category by ID
         * @method GET
         * @param path - /categories/:id
         */
        this.router.get(
            '/:id',
            this.controller.getCategoryById
        );

        // Apply the auth middlewares
        this.router.use(
            this.authMiddleware
        );

        /**
         * Authenticated users with the correct permissions can manipulate categories
        **/

        /**
         * Creates a new category
         * @method POST
         * @param path - /categories
         */
        this.router.post(
            '/',
            this.controller.createCategory
        );

        /**
         * Updates a category by ID
         * @method PUT
         * @param path - /categories/:id
         */
        this.router.put(
            '/:id',
            this.controller.updateCategory
        );

        /**
         * Deletes a category by ID
         * @method DELETE
         * @param path - /categories/:id
         */
        this.router.delete(
            '/:id',
            this.controller.deleteCategory
        );
    }

    /**
     * Get the router with all routes
     * @returns Express Router instance
     */
    public getRoutes(): Router {
        return this.router;
    }
}