import { RequestHandler, Router } from "express";
import { IRouter } from "../../shared/contracts/IRouter";
import { DisciplineController } from "../../modules/Discipline/discipline.controller";

/**
 * Router class for discipline endpoints.
 * Maps HTTP routes to DisciplineController methods.
 */
export class DisciplineRouter implements IRouter {
    path = '/admin/disciplines';
    router: Router;
    private controller: DisciplineController;
    private authMiddleware: RequestHandler;
    private permissionsMiddleware: RequestHandler;

    constructor (
        controller: DisciplineController,
        authMiddleware: RequestHandler,
        permissionsMiddleware: RequestHandler,
    ) {
        this.controller = controller;
        this.authMiddleware = authMiddleware;
        this.permissionsMiddleware = permissionsMiddleware;
        this.router = Router();
        this.initRoutes();
    }

    /**
     * Initializes all routes for the discipline controller
     * Maps endpoints to controller methods
     */
    initRoutes() {
        /**
         * List all disciplines
         * @method GET
         * @param path - /disciplines
         */
        this.router.get(
            '/',
            this.controller.listDisciplines
        );

        /**
         * Finds a discipline by ID
         * @method GET
         * @param path - /disciplines/:id
         */
        this.router.get(
            '/:id',
            this.controller.getDisciplineById
        );

        // Apply the auth middlewares
        this.router.use(
            this.authMiddleware
        );

        /**
         * Authenticated users with the correct permissions can manipulate disciplines
        **/

        /**
         * Creates a new a discipline
         * @method POST
         * @param path - /disciplines
         */
        this.router.post(
            '/:id',
            this.controller.createDiscipline
        );

        /**
         * Update a discipline by ID
         * @method PUT
         * @param path - /disciplines/:id
         */
        this.router.put(
            '/:id',
            this.controller.updateDiscipline
        );

        /**
         * Deletes a discipline by ID
         * @method DELETE
         * @param path - /disciplines/:id
         */
        this.router.delete(
            '/:id',
            this.controller.deleteDiscipline
        );
    }

    /**
     * Get the router with all routes
     * @returns Express Router instance
     */
    public getRoutes(): Router {
        return this.router;
    }
}