import { RequestHandler, Router } from "express";
import { IRouter } from "../../shared/contracts/IRouter";
import { StudentController } from "../../modules/Student/student.controller";

/**
 * Router class for students endpoints.
 * Maps HTTP routes to StudentController methods.
 */
export class StudentRouter implements IRouter {
    path = '/app/students';
    router: Router;
    private controller: StudentController;
    private authRouter: IRouter;
    private authMiddleware: RequestHandler;

    constructor (
        controller: StudentController,
        authRouter: IRouter,
        authMiddleware: RequestHandler,
    ) {
        this.controller = controller;
        this.authRouter = authRouter;
        this.authMiddleware = authMiddleware;
        this.router = Router();
        this.initRoutes();
    }

    /**
     * Initializes all routes for the student router
     * Maps endpoints to controller methods
     */
    initRoutes() {
        /**
         * Apply the authentication routes
         */
        this.router.use(
            this.authRouter.path,
            this.authRouter.getRoutes()
        );

        /*
         * Apply the auth middleware
         * Only authenticated students
         */
        this.router.use(
            this.authMiddleware
        );

        /**
         * Lists all students
         * @method GET
         * @param path - /students
         */
        this.router.get(
            '/',
            this.controller.listAdaptedStudents
        );

        /**
         * Find teachers from students
         * @method GET
         * @param path - /students/teachers
         */
        this.router.get(
            '/teachers',
            this.controller.getTeachers
        );

        /**
         * Create scheduled class
         * @method POST
         * @param path - /students/classes
         */
        this.router.post(
            '/classes',
            this.controller.createClass
        );

        /**
         * List all student's classes
         * @method GET
         * @param path - /students/classes
         */
        this.router.get(
            '/classes',
            this.controller.getClasses
        );

        /**
         * Get one class by ID
         * @method GET
         * @param path - /students/classes/:id
         */
        this.router.get(
            '/classes/:id',
            this.controller.getClassById
        );

        /**
         * Updates a class by ID
         * @method PUT
         * @param path - /students/classes/:id
         */
        this.router.put(
            '/classes/:id',
            this.controller.updateClass
        );

        /**
         * Cancel a class by ID
         * @method PUT
         * @param path - /students/classes/:id/cancel
         */
        this.router.put(
            '/classes/:id/cancel',
            this.controller.cancelClass
        );
    }

    /**
     * Get the router with all routes
     * @returns Express Router instance
     */
    public getRoutes(): Router {
        return this.router;
    }
}