import { TeacherController } from "../../modules/Teacher/teacher.controller";
import { IRouter } from "../../shared/contracts/IRouter";
import { Router, RequestHandler } from "express";

/**
 * Router class for teachers endpoints.
 * Maps HTTP routes to TeacherController methods.
 */
export class TeacherRouter implements IRouter {
    path = "/app/teachers";
    router: Router;
    private controller: TeacherController;
    private authRouter: IRouter;
    private authMiddleware: RequestHandler;

    constructor (
        controller: TeacherController,
        authRouter: IRouter,
        authMiddleware: RequestHandler,
    ) {
        this.controller = controller;
        this.authRouter = authRouter;
        this.authMiddleware = authMiddleware;
        this.router = Router();
        this.initRoutes();
    }

    /**
     * Initializes all routes for the teacher router
     * Maps endpoints to controller methods
     */
    initRoutes() {
        /**
         * Apply the authentication routes
         */
        this.router.use(
            this.authRouter.path,
            this.authRouter.getRoutes()
        );

        /**
         * Apply the auth middleware
         */
        this.router.use(this.authMiddleware);

        /**
         * Creates a new teacher
         * @method POST
         * @param path - /teachers
         */
        this.router.post(
            "/",
            this.controller.createTeacher
        );

        /**
         * List all teachers
         * @method GET
         * @param path - /teachers
         */
        this.router.get(
            "/",
            this.controller.listTeachers
        );

        /**
         * List teacher's classes
         * @method GET
         * @param path - /teachers/classes
         */
        this.router.get(
            "/classes",
            this.controller.getClasses
        );

        /**
         * Find one class by ID
         * @method GET
         * @param path - /teachers/classes/:id
         */
        this.router.get(
            "/classes/:id",
            this.controller.getClassById
        );

        /**
         * Finish one class by ID
         * @method GET
         * @param path - /teachers/classes/:id/finish
         */
        this.router.post(
            "/classes/:id/finish",
            this.controller.finishClass
        );
    }

    /**
     * Get the router with all routes
     * @returns Express Router instance
     */
    public getRoutes(): Router {
        return this.router;
    }
}