import { RequestHandler, Router } from "express";
import { IRouter } from "../../shared/contracts/IRouter";
import { TopicController } from "../../modules/Topic/topic.controller";

/**
 * Router class for topic endpoints.
 * Maps HTTP routes to TopicController methods.
 */
export class TopicRouter implements IRouter {
    path = '/admin/topics';
    router: Router;
    private controller: TopicController;
    authMiddleware: RequestHandler;
    permissionMiddleware: RequestHandler;

    constructor (
        controller: TopicController,
        authMiddleware: RequestHandler,
        permissionsMiddleware: RequestHandler,
    ) {
        this.controller = controller;
        this.authMiddleware = authMiddleware;
        this.permissionMiddleware = permissionsMiddleware;
        this.router = Router();
        this.initRoutes();
    }

    /**
     * Initializes all routes for the topic controller
     * Maps endpoints to controller methods
     */
    initRoutes() {

        /**
         * Unprotected routes
         * Required for registration
        **/

        /**
         * Finds a topic by ID
         * @method GET
         * @param path - /topics
         */
        this.router.get(
            '/',
            this.controller.listTopics
        );

        /**
         * Finds a topic by ID
         * @method GET
         * @param path - /topics
         */
        this.router.get(
            '/:id',
            this.controller.getTopicById
        );

        // Apply the auth middlewares
        this.router.use(
            this.authMiddleware
        );

        /**
         * Authenticated users with the correct permissions can manipulate topics
        **/

        /**
         * Creates a new topic
         * @method POST
         * @param path - /topics
         */
        this.router.post(
            '/',
            this.controller.createTopic
        );

        /**
         * Updates a topic by ID
         * @method PUT
         * @param path - /topics/:id
         */
        this.router.put(
            '/:id',
            this.controller.updateTopic
        );

        /**
         * Deletes a topic by ID
         * @method DELETE
         * @param path - /topics/:id
         */
        this.router.delete(
            '/:id',
            this.controller.deleteTopic
        );
    }

    /**
     * Get the router with all routes
     * @returns Express Router instance
     */
    public getRoutes(): Router {
        return this.router;
    }
}