import { Router, RequestHandler } from 'express';
import { IRouter } from '../../shared/contracts/IRouter';
import { UserController } from '../../modules/user/user.controller';

/**
 * Router class for users endpoints.
 * Maps HTTP routes to UserController methods.
 */
export class UserRouter implements IRouter {
    path = '/admin/users';
    router: Router;
    authRouter: IRouter;
    private controller: UserController;
    private authMiddleware: RequestHandler;
    private permissionsMiddleware: RequestHandler;
    private classesRouter: IRouter;

    constructor (
        controller: UserController,
        authRouter: IRouter,
        classesRouter: IRouter,
        authMiddleware: RequestHandler,
        permissionsMiddleware: RequestHandler,
    ) {
        this.router = Router();
        this.authRouter = authRouter;
        this.classesRouter = classesRouter;
        this.authMiddleware = authMiddleware;
        this.permissionsMiddleware = permissionsMiddleware;
        this.controller = controller;
        this.initRoutes();
    }

    /**
     * Initializes all routes for the user router.
     * Maps endpoints to user controller methods.
     */
    initRoutes() {
        /**
         * Apply the authentication router
         */
        this.router.use(
            this.authRouter.path,
            this.authRouter.getRoutes()
        );

        /**
         * Apply the authentication middleware to all routes
         */
        this.router.use(this.authMiddleware);

        /**
         * Apply the routes for classes CRUD
         */
        this.router.use(
            '/classes',
            this.classesRouter.getRoutes()
        );

        this.router.post(
            '/students',
            this.controller.createStudent
        );

        this.router.get(
            '/students',
            this.controller.listStudents
        );

        this.router.get(
            '/students/:id',
            this.controller.getStudentById
        );

        this.router.put(
            '/students/:id',
            this.controller.updateStudent
        );

        this.router.delete(
            '/students/:id',
            this.controller.deleteStudent
        );

        this.router.post(
            '/teachers',
            this.controller.createTeacher
        );

        this.router.get(
            '/teachers',
            this.controller.listTeachers
        );

        this.router.get(
            '/teachers/:id',
            this.controller.getTeacherById
        );

        this.router.get(
            '/teachers/:id',
            this.controller.updateTeacher
        );

        this.router.get(
            '/teachers/:id',
            this.controller.deleteTeacher
        );

        /**
         * POST /users - Create a new user
         */
        this.router.post(
            "/",
            this.controller.createUser
        );

        /**
         * GET /users - Get all users
         */
        this.router.get(
            "/",
            this.controller.getUsers
        );

        /**
         * GET /users/:id - Get a user by ID
         */
        this.router.get(
            "/:id",
            this.controller.getUserById
        );

        /**
         * PUT /users/:id - Update a user by ID
         */
        this.router.put(
            "/:id",
            this.controller.updateUser
        );

        /**
         * DELETE /users/:id - Delete a user by ID
         */
        this.router.delete(
            "/:id",
            this.controller.deleteUser
        );

        /**
         * GET /users/:id/roles - Get all roles for a user by ID
         */
        this.router.get(
            "/:id/roles",
            this.controller.getUserRolesById
        );

        /**
         * POST /users/:id/roles - Assign roles to a user by ID
         */
        this.router.post(
            "/:id/roles",
            this.controller.assignRoleToUser
        );

        /**
         * DELETE /users/:id/roles - Remove roles from a user by ID
         */
        this.router.delete(
            "/:id/roles",
            this.controller.removeRoleFromUser
        );
    }

    /**
     * Get the router with all routes
     */
    public getRoutes(): Router {
        return this.router;
    }
}