import { z } from 'zod';
import { emailValidation } from './shared/email.schema';
import { passwordValidation } from './shared/password.schema';
import { userSchemas } from './user.schema';
import { studentSchemas } from './student.schema';
import { teacherSchemas } from './teacher.schema';

const loginBodySchema = z.object({
    email: emailValidation,
    password: passwordValidation,
});

const registerBodySchema = z.object({
    email: emailValidation,
    password: passwordValidation,
}).passthrough();

const forgotPasswordBodySchema = z.object({
    email: emailValidation,
});

const resetPasswordBodySchema = z.object({
    email: emailValidation,
    code: z
        .string()
        .trim()
        .length(6, 'Verification code must have 6 characters.'),
    newPassword: passwordValidation,
});

/**
 * Collection of schemas responsible for validating authentication requests.
 */
const authSchemas = {
    register: z.object({
        body: registerBodySchema,
    }),
    login: z.object({
        body: loginBodySchema,
    }),
    forgotPassword: z.object({
        body: forgotPasswordBodySchema,
    }),
    resetPassword: z.object({
        body: resetPasswordBodySchema,
    }),
};

/**
 * Schemas for specific entity registration and profile updates.
 * These leverage existing entity-specific schemas.
 */
export const allAuthSchemas = {

    userAuthSchemas: {
        ...authSchemas,
        register: userSchemas.createUser,
        update: z.object({
            body: userSchemas.updateUser.shape.body,
        }),
    },

    // Student
    studentAuthSchemas: {
        ...authSchemas,
        register: studentSchemas.createStudent,
        update: z.object({
            body: studentSchemas.updateStudent.shape.body,
        }),
    },

    // Teacher
    teacherAuthSchemas: {
        ...authSchemas,
        register: teacherSchemas.createTeacher,
        update: z.object({
            body: teacherSchemas.updateTeacher.shape.body,
        }),
    },
}

export type IAuthSchemas =
    typeof allAuthSchemas.studentAuthSchemas |
    typeof allAuthSchemas.teacherAuthSchemas |
    typeof allAuthSchemas.userAuthSchemas
;

export type IUpdateEntity =
    typeof allAuthSchemas.studentAuthSchemas.update |
    typeof allAuthSchemas.teacherAuthSchemas.update |
    typeof allAuthSchemas.userAuthSchemas.update
;