import { z } from 'zod';
import { uuidV7Schema } from './shared/uuidv7.schema';

/**
 * Shared fields for category-related payloads
 */
export const categorySchema = {
    name: z
        .string()
        .trim()
        .min(1, 'Category name cannot be empty.')
        .max(100, 'Category name cannot exceed 100 characters.')
        .regex(
            /^[a-zA-Z0-9À-ÿ\s\-_&]+$/,
            'Category name can only contain letters, numbers, spaces, hyphens, underscores, and ampersands.'
        )
        .refine(
            (name) => {
                // Check for consecutive special characters
                return !/(\s{2,}|_{2,}|-{2,})/.test(name);
            },
            {
                message: 'Category name cannot contain consecutive special characters or spaces.',
            }
        ),

    description: z
        .string()
        .trim()
        .min(1, 'Category description cannot be empty.')
        .max(500, 'Category description cannot exceed 500 characters.')
        .optional(),
};

/**
 * Schema for creating a new category
 */
const createCategoryBodySchema = z.object({
    name: categorySchema.name,
    description: categorySchema.description,
});

/**
 * Schema for updating an existing category
 */
const updateCategoryBodySchema = z
    .object({
        name: categorySchema.name.optional(),
        description: categorySchema.description,
    })
    .superRefine((data, ctx) => {
        if (Object.keys(data).length === 0) {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'Provide at least one field to update.',
                path: [],
            });
        }

        // Validate that if description is provided, it's not empty
        if (data.description !== undefined && data.description.trim() === '') {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'Description cannot be empty if provided.',
                path: ['description'],
            });
        }

        // Validate that name doesn't contain only special characters when provided
        if (data.name !== undefined) {
            const nameWithoutSpecialChars = data.name.replace(/[^a-zA-Z0-9À-ÿ]/g, '');
            if (nameWithoutSpecialChars.length === 0) {
                ctx.addIssue({
                    code: z.ZodIssueCode.custom,
                    message: 'Category name must contain at least one letter or number.',
                    path: ['name'],
                });
            }
        }
    });

/**
 * Collection of schemas responsible for validating all category module requests.
 */
export const categorySchemas = {
    createCategory: z.object({
        body: createCategoryBodySchema,
    }),
    getCategoryById: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
    updateCategory: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
        body: updateCategoryBodySchema,
    }),
    deleteCategory: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
};

/**
 * Type representing the category schemas structure for dependency injection
 */
export type ICategorySchemas = typeof categorySchemas;

/**
 * Schema that validates a non-empty list of category IDs for bulk operations
 */
export const categoryIdsSchema = z
    .array(uuidV7Schema)
    .min(1, 'Provide at least one category ID')
    .max(100, 'Cannot process more than 100 categories at once');

/**
 * Schema for category name uniqueness check
 */
export const categoryNameSchema = z.object({
    name: categorySchema.name,
    exclude_id: uuidV7Schema.optional(), // For update operations, exclude current category
});