import { z } from 'zod';
import { uuidV7Schema } from './shared/uuidv7.schema';
import { stringSchemas } from './shared/simpleString.schema';
import { dateValidation } from './shared/date.schema';

/**
 * Shared fields for class-related payloads
 */
const classSchema = {
    topic_id: uuidV7Schema,

    teacher_id: uuidV7Schema,

    main_student_id: uuidV7Schema,

    additional_students: z
        .array(uuidV7Schema)
        .max(20, 'Cannot exceed 20 additional students.')
        .optional(),

    link: stringSchemas.url.optional(),

    type: z.enum(['individual', 'group']),

    start_time: dateValidation.isoDateSchema
        .transform(val => new Date(val))
        .pipe(z.date()),

    end_time: dateValidation.isoDateSchema
        .transform(val => new Date(val))
        .pipe(z.date()),

    status: z.enum(['scheduled', 'canceled']),

    payment_status: z.enum(['pending', 'paid', 'refunded']),
};

/**
 * Schema for creating a new class
 */
const createClassBodySchema = z
    .object({
        topic_id: classSchema.topic_id,
        teacher_id: classSchema.teacher_id,
        main_student_id: classSchema.main_student_id,
        additional_students: classSchema.additional_students.optional(),
        start_time: classSchema.start_time,
        end_time: classSchema.end_time.optional(),
    })
    .superRefine((data, ctx) => {
        // Validate that end_time > start_time
        if (data.end_time && data.end_time <= data.start_time) {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'end_time must be after start_time.',
                path: ['end_time'],
            });
        }
    });

/**
 * Schema for updating an existing class
 */
const updateClassBodySchema = z
    .object({
        topic_id: classSchema.topic_id.optional(),
        teacher_id: classSchema.teacher_id.optional(),
        additional_students: classSchema.additional_students.optional(),
        start_time: classSchema.start_time.optional(),
        end_time: classSchema.end_time.optional(),
        status: classSchema.status.optional(),
        payment_status: classSchema.payment_status.optional(),
    })
    .superRefine((data, ctx) => {
        if (Object.keys(data).length === 0) {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'Provide at least one field to update.',
                path: [],
            });
        }

        // Check end_time > start_time if both exist
        if (data.start_time && data.end_time && data.end_time <= data.start_time) {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'end_time must be after start_time.',
                path: ['end_time'],
            });
        }
    });

/**
 * Query parameters for filtering classes
 */
export const getClassesQuerySchema = z.object({
    topic_id: uuidV7Schema.optional(),
    teacher_id: uuidV7Schema.optional(),
    student_id: uuidV7Schema.optional(),
    type: z.enum(['individual', 'group']).optional(),
    status: z.enum(['scheduled', 'canceled', 'completed']).optional(),
    payment_status: z.enum(['pending', 'paid', 'refunded']).optional(),
});

/**
 * Collection of schemas responsible for validating all class module requests.
 */
export const classSchemas = {
    createClass: z.object({
        body: createClassBodySchema,
    }),

    getClassById: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),

    getClasses: z.object({
        query: getClassesQuerySchema,
    }),

    updateClass: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
        body: updateClassBodySchema,
    }),

    deleteClass: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
};

/**
 * Type for class schemas dependency injection
 */
export type IClassSchemas = typeof classSchemas;