import { z } from 'zod';
import { uuidV7Schema } from './shared/uuidv7.schema';

/**
 * Shared fields for role-related payloads.
 */
export const roleSchema = {
    name: z
        .string()
        .trim()
        .min(1, 'Role name cannot be empty.')
        .max(100, 'Role name cannot exceed 100 characters.')
        .regex(
            /^[a-zA-Z0-9_\- ]+$/,
            'Role name can only contain letters, numbers, spaces, hyphens and underscores.'
        ),
    description: z
        .string()
        .trim()
        .min(1, 'Role description cannot be empty.')
        .max(500, 'Role description cannot exceed 500 characters.')
};

/**
 * Schema for creating a new role
 */
const createRoleBodySchema = z.object({
    name: roleSchema.name,
    description: roleSchema.description,
});

/**
 * Schema for updating an existing role
 */
const updateRoleBodySchema = z
    .object({
        name: roleSchema.name.optional(),
        description: roleSchema.description.optional(),
    })
    .superRefine((data, ctx) => {
        if (Object.keys(data).length === 0) {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'Provide at least one field to update.',
                path: [],
            });
        }

        // Validate that if description is provided, it's not empty
        if (data.description !== undefined && data.description.trim() === '') {
            ctx.addIssue({
                code: z.ZodIssueCode.custom,
                message: 'Description cannot be empty if provided.',
                path: ['description'],
            });
        }
    });

/**
 * Collection of schemas responsible for validating all role module requests.
 */
export const roleSchemas = {
    createRole: z.object({
        body: createRoleBodySchema,
    }),
    getRoleById: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
    updateRole: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
        body: updateRoleBodySchema,
    }),
    deleteRoleById: z.object({
        params: z.object({
            id: uuidV7Schema,
        }),
    }),
};

/**
 * Type for role schemas dependency injection
 */
export type IRoleSchemas = typeof roleSchemas;